package com.atlassian.plugins.navlink.producer.contentlinks.customcontentlink;

import com.atlassian.activeobjects.external.ActiveObjects;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugins.navlink.producer.contentlinks.plugin.CustomContentLinkProviderModuleDescriptor;
import com.atlassian.plugins.navlink.spi.Project;
import com.atlassian.plugins.navlink.spi.ProjectManager;
import com.atlassian.plugins.navlink.spi.ProjectNotFoundException;
import com.atlassian.plugins.navlink.spi.ProjectPermissionManager;
import com.atlassian.sal.api.transaction.TransactionCallback;
import com.atlassian.sal.api.user.UserManager;
import com.google.common.base.Function;
import com.google.common.collect.Lists;

import net.java.ao.Query;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

public class DefaultCustomContentLinkService implements CustomContentLinkService
{
    private static final Logger log = LoggerFactory.getLogger(DefaultCustomContentLinkService.class);

    private ActiveObjects ao;
    private final ProjectManager projectManager;
    private final ProjectPermissionManager projectPermissionManager;
    private final UserManager userManager;

    private PluginAccessor pluginAccessor;

    final Function<CustomContentLinkAO, CustomContentLink> aoToEntityTransformer = new Function<CustomContentLinkAO, CustomContentLink>()
    {
        @Override
        public CustomContentLink apply(@Nullable CustomContentLinkAO input) {
            return CustomContentLink.builder().id(input.getID()).url(input.getLinkUrl()).label(input.getLinkLabel())
                    .key(input.getContentKey()).sequence(input.getSequence()).build();
        }
    };

    public DefaultCustomContentLinkService(ActiveObjects ao, PluginAccessor pluginAccessor, ProjectManager projectManager, ProjectPermissionManager projectPermissionManager, UserManager userManager)
    {
        this.ao = ao;
        this.pluginAccessor = pluginAccessor;
        this.projectManager = projectManager;
        this.projectPermissionManager = projectPermissionManager;
        this.userManager = userManager;
    }

    @Override
    public CustomContentLink addCustomContentLink(final CustomContentLink customContentLink) throws NoAdminPermissionException {
        checkPermission(customContentLink.getContentKey());
        return aoToEntityTransformer.apply(ao.executeInTransaction(new TransactionCallback<CustomContentLinkAO>() {
            @Override
            public CustomContentLinkAO doInTransaction()
            {
                CustomContentLinkAO link = ao.create(CustomContentLinkAO.class, Collections.<String, Object> emptyMap());
                link.setContentKey(customContentLink.getContentKey());
                link.setLinkLabel(customContentLink.getLinkLabel());
                link.setLinkUrl(customContentLink.getLinkUrl());
                link.setSequence(getNextSequence(customContentLink.getContentKey()));
                link.save();
                return link;
            }
        }));
    }

    private void copyToAO(CustomContentLink customContentLink, CustomContentLinkAO link) {
        link.setContentKey(customContentLink.getContentKey());
        link.setLinkLabel(customContentLink.getLinkLabel());
        link.setLinkUrl(customContentLink.getLinkUrl());
    }

    private void checkPermission(String contentKey) throws NoAdminPermissionException {
        String userName = userManager.getRemoteUsername();
        try {
            Project p = projectManager.getProjectByKey(contentKey);
            if (!projectPermissionManager.canAdminister(p, userName)) {
                throw new NoAdminPermissionException(userName, contentKey, null);
            }
        } catch (ProjectNotFoundException e) {
            throw new NoAdminPermissionException(userName, contentKey, e);
        }
    }

    @Override
    public List<CustomContentLink> getCustomContentLinks(final String key)
    {
        List<CustomContentLink> links = ao.executeInTransaction(new TransactionCallback<List<CustomContentLink>>()
        {
            @Override
            public List<CustomContentLink> doInTransaction() {
                return Lists.newArrayList(Lists.transform(
                        getAOsByKey(key),
                        aoToEntityTransformer));
            }
        });
        return links;
    }

    private List<CustomContentLinkAO> getAOsByKey(String key) {
        return Arrays.asList(ao.find(CustomContentLinkAO.class, Query.select().where("CONTENT_KEY = ?", key).order("SEQUENCE asc")));
    }

    public List<CustomContentLink> getPluginCustomContentLinks(String key)
    {
        List<CustomContentLink> links = new ArrayList<CustomContentLink>();
        List<CustomContentLinkProviderModuleDescriptor> moduleDescriptors = pluginAccessor
                .getEnabledModuleDescriptorsByClass(CustomContentLinkProviderModuleDescriptor.class);
        for (CustomContentLinkProviderModuleDescriptor descriptor : moduleDescriptors)
        {
            try
            {
                links.addAll(descriptor.getModule().getCustomContentLinks(key));
            }
            catch (Exception ex)
            {
                log.warn(
                        "Error getting custom content links using CustomContentLinkProviderModule: {} with content link key: {}",
                        descriptor.getModule(), key);
                log.debug("Stack trace:", ex);
            }
        }
        return links;
    }

    private CustomContentLinkAO[] getMatchingCustomContentLink(CustomContentLink entity) {
        return ao.find(
                CustomContentLinkAO.class,
                Query.select().where("CONTENT_KEY = ? AND LINK_URL = ? and LINK_LABEL = ?", entity.getContentKey(),
                        entity.getLinkUrl(), entity.getLinkLabel()));
    }

    @Override
    public void removeCustomContentLink(final CustomContentLink customContentLink) throws NoAdminPermissionException {
        checkPermission(customContentLink.getContentKey());
        ao.executeInTransaction(new TransactionCallback<Boolean>() {
            public Boolean doInTransaction() {
                try {
                    CustomContentLinkAO[] results = getMatchingCustomContentLink(customContentLink);
                    if (results.length > 0) {
                        ao.delete(results[0]);
                        return true;
                    }
                    return false;
                } finally {
                    reSequence(customContentLink.getContentKey());
                }
            }
        });
    }

    @Override
    public CustomContentLink getById(int id) {
        CustomContentLinkAO link = getAOById(id);
        return link == null ? null : aoToEntityTransformer.apply(link);
    }

    @Override
    public void removeById(final int id) throws NoAdminPermissionException {
        final CustomContentLinkAO link = getAOById(id);

        if (link != null) { //TODO what's the behaviour of get if nothing is found?
            checkPermission(link.getContentKey());
            ao.executeInTransaction(new TransactionCallback<Void>() {
                @Override
                public Void doInTransaction() {
                    ao.delete(link);
                    reSequence(link.getContentKey());
                    return null;
                }
            });
        }
    }

    @Override
    public void update(final CustomContentLink newValue) throws NoAdminPermissionException {
        final CustomContentLinkAO link = getAOById(newValue.getId());

        if (link != null) { //TODO what's the behaviour of get if nothing is found?
            checkPermission(link.getContentKey());
            ao.executeInTransaction(new TransactionCallback<Void>() {
                @Override
                public Void doInTransaction() {
                    copyToAO(newValue, link);
                    link.save();
                    return null;
                }
            });
        }
    }

    @Override
    public void moveAfter(int idToMove, int idToComeAfter) throws NoAdminPermissionException {
        CustomContentLinkAO linkToMove = getAOById(idToMove);
        CustomContentLinkAO linkToComeAfter = getAOById(idToComeAfter);
        if (!linkToComeAfter.getContentKey().equals(linkToMove.getContentKey())) {
            throw new IllegalArgumentException("Tried to move link " + linkToMove + " after " + linkToComeAfter + ", content keys differ.");
        }
        moveToIndex(linkToMove, linkToComeAfter.getSequence() + 1);
    }

    private void moveToIndex(CustomContentLinkAO linkToMove, int indexToMoveTo) throws NoAdminPermissionException {
        checkPermission(linkToMove.getContentKey());
        List<CustomContentLinkAO> links = new ArrayList<CustomContentLinkAO>(getAOsByKey(linkToMove.getContentKey())); // needs to be a mutable list
        int indexOfMovingLink = links.indexOf(linkToMove);
        if (indexToMoveTo > indexOfMovingLink) {
            indexToMoveTo -= 1;
        }
        links.remove(linkToMove);
        if (indexToMoveTo >= links.size()) { // we are moving to the end
            links.add(linkToMove);
        } else {

            links.add(indexToMoveTo, linkToMove);
        }
        reSequenceList(links);
    }

    private void reSequenceList(final List<CustomContentLinkAO> links) {
        int sequence = 0;
        for (CustomContentLinkAO link : links) {
            link.setSequence(sequence++);
            link.save();
        }
    }

    @Override
    public void moveToStart(int id) throws NoAdminPermissionException {
        moveToIndex(getAOById(id), 0);
    }


    private void reSequence(String contentKey) {
        reSequenceList(getAOsByKey(contentKey));
    }

    private int getNextSequence(String key) {
        List<CustomContentLinkAO> links = getAOsByKey(key);
        Integer maxSequence = null;
        for (CustomContentLinkAO link : links) {
            if (maxSequence == null || link.getSequence() > maxSequence) {
                maxSequence = link.getSequence();
            }
        }
        return maxSequence == null ? 0 : maxSequence + 1;
    }

    private CustomContentLinkAO getAOById(int id) {
        return ao.get(CustomContentLinkAO.class, id);
    }
}
