package com.atlassian.plugins.navlink.producer.capabilities.services;

import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugins.navlink.producer.capabilities.ApplicationWithCapabilities;
import com.atlassian.plugins.navlink.producer.capabilities.Capability;
import com.atlassian.plugins.navlink.producer.capabilities.plugin.CapabilityModuleDescriptor;
import com.atlassian.sal.api.ApplicationProperties;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Default implementation of the {@link CapabilityService}
 *
 * @since 1.0
 */
public class DefaultCapabilityService implements CapabilityService
{
    private final Logger logger = LoggerFactory.getLogger(DefaultCapabilityService.class);

    private final PluginAccessor pluginAccessor;
    private final ApplicationTypeService applicationTypeService;
    private final ApplicationProperties applicationProperties;

    public DefaultCapabilityService(PluginAccessor pluginAccessor, ApplicationTypeService applicationTypeService,
                                    ApplicationProperties applicationProperties)
    {
        this.pluginAccessor = pluginAccessor;
        this.applicationTypeService = applicationTypeService;
        this.applicationProperties = applicationProperties;
    }

    @Nonnull
    @Override
    public ApplicationWithCapabilities getHostApplication()
    {
        final String applicationType = applicationTypeService.get();
        final List<Capability> capabilitiesFromPluginModules = getCapabilitiesFromPluginModules();
        final Iterable<Capability> usableCapabilities = Iterables.filter(capabilitiesFromPluginModules, byApplicationType(applicationType));
        final Map<String, String> capabilities = asMap(usableCapabilities);

        return new ApplicationWithCapabilities(applicationType, new DateTime(applicationProperties.getBuildDate()), capabilities);
    }

    @Nonnull
    private List<Capability> getCapabilitiesFromPluginModules()
    {
        final List<CapabilityModuleDescriptor> enabledModuleDescriptors = getEnabledModuleDescriptors();
        return Lists.newArrayList(Iterables.transform(enabledModuleDescriptors, toCapability()));
    }

    @Nonnull
    private List<CapabilityModuleDescriptor> getEnabledModuleDescriptors()
    {
        final List<CapabilityModuleDescriptor> moduleDescriptors = pluginAccessor.getEnabledModuleDescriptorsByClass(CapabilityModuleDescriptor.class);
        return moduleDescriptors != null ? moduleDescriptors : Collections.<CapabilityModuleDescriptor>emptyList();
    }

    @Nonnull
    private Function<CapabilityModuleDescriptor, Capability> toCapability()
    {
        return new Function<CapabilityModuleDescriptor, Capability>()
        {
            @Nullable
            @Override
            public Capability apply(@Nullable final CapabilityModuleDescriptor input)
            {
                return input != null ? input.getModule() : null;
            }
        };
    }

    @Nonnull
    private Predicate<Capability> byApplicationType(@Nonnull final String applicationType)
    {
        return new Predicate<Capability>()
        {
            @Override
            public boolean apply(@Nullable Capability input)
            {
                if (input != null)
                {
                    final String type = input.getType();
                    return Strings.isNullOrEmpty(type) || applicationType.equals(type);
                }
                else
                {
                    return false;
                }
            }

        };
    }

    @Nonnull
    private Map<String, String> asMap(@Nonnull final Iterable<Capability> usableCapabilities)
    {
        final Map<String, String> capabilities = new HashMap<String, String>();
        for (final Capability capability : usableCapabilities)
        {
            logger.debug("Capability: {} at URL {}", new Object[]{capability.getName(), capability.getUrl()});
            capabilities.put(capability.getName(), capability.getUrl());
        }
        return capabilities;
    }
}
