package com.atlassian.plugins.navlink.producer.capabilities.plugin;

import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginParseException;
import com.atlassian.plugin.descriptors.AbstractModuleDescriptor;
import com.atlassian.plugin.module.ModuleFactory;
import com.atlassian.plugin.util.validation.ValidationPattern;
import com.atlassian.plugins.navlink.producer.capabilities.Capability;
import com.atlassian.util.concurrent.NotNull;
import com.google.common.base.Strings;
import org.dom4j.Attribute;
import org.dom4j.Element;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static com.atlassian.plugin.util.validation.ValidationPattern.test;
import static com.google.common.base.Preconditions.checkNotNull;

public class CapabilityModuleDescriptor extends AbstractModuleDescriptor<Capability>
{
    private Element element;
    private boolean enabled;

    public CapabilityModuleDescriptor(@Nonnull final ModuleFactory moduleFactory)
    {
        super(checkNotNull(moduleFactory));
    }

    @Override
    public void init(@NotNull final Plugin plugin, @NotNull final Element element) throws PluginParseException
    {
        super.init(checkNotNull(plugin), checkNotNull(element));
        this.element = element;
        this.enabled = false;
    }

    @Override
    protected void provideValidationRules(ValidationPattern pattern)
    {
        super.provideValidationRules(pattern);
        pattern.rule(
                test("name").withError("name tag is mandatory"),
                test("string-length(normalize-space(name)) > 0").withError("name tag requires a name as content"),
                test("url").withError("url tag is mandatory"),
                test("string-length(normalize-space(url)) > 0").withError("url tag requires a url as content"),
                test("not(@type) or string-length(normalize-space(@type)) > 0").withError("type attribute of the capability tag is empty")
        );
    }

    @Override
    public void enabled()
    {
        super.enabled();
        enabled = true;
    }

    @Override
    public void disabled()
    {
        super.disabled();
        enabled = false;
    }

    @Override
    public Capability getModule()
    {
        return enabled ? parseCapability() : null;
    }

    @Nonnull
    private Capability parseCapability()
    {
        final String type = parseType();
        final String name = parseName();
        final String url = parseUrl();
        return new Capability(type, name, url);
    }

    @Nonnull
    private String parseType()
    {
        final Attribute typeAttribute = element.attribute("type");
        return typeAttribute != null ? typeAttribute.getValue().trim() : "";
    }

    @Nonnull
    private String parseName()
    {
        return parseElementContent(element.element("name"));
    }

    @Nonnull
    private String parseUrl()
    {
        return parseElementContent(element.element("url"));
    }

    @Nonnull
    private String parseElementContent(@Nullable final Element element)
    {
        final String content = element != null ? element.getTextTrim().trim() : null;
        return Strings.nullToEmpty(content);
    }
}
