package com.atlassian.plugins.navlink.producer.capabilities;

import com.google.common.base.Objects;
import com.google.common.base.Strings;

import javax.annotation.concurrent.Immutable;

/**
 * A capability that the host application provides.
 *
 * @since 1.0
 */
@Immutable
public class Capability
{
    private String type;
    private String name;
    private String url;

    /**
     * A capability to be used by applications.
     *
     * @param type the application type this capability can be used with; in case the type is <code>null</code> or empty
     *             the capability can be used in any application
     * @param name the name of the capability
     * @param url the REST endpoint of the capability
     */
    public Capability(final String type, final String name, final String url)
    {
        this.type = Strings.nullToEmpty(type);
        this.name = name;
        this.url = url;
    }

    /**
     * Limit the usage of the capability to the specified application type only. If empty, the capability can be used in
     * any application.
     *
     * @return the application type the capability can be used with
     */
    public String getType()
    {
        return type;
    }

    /**
     * Gets the name of this capability, corresponding to one defined in the Capability API specification.
     *
     * @return the name
     */
    public String getName()
    {
        return name;
    }

    /**
     * Gets the URL for the REST resource defining this capability.
     *
     * @return the URL
     */
    public String getUrl()
    {
        return url;
    }

    @Override
    public int hashCode()
    {
        return Objects.hashCode(type, name, url);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (obj == this)
        {
            return true;
        }

        if (!(obj instanceof Capability))
        {
            return false;
        }

        final Capability that = (Capability) obj;
        return Objects.equal(this.type, that.type) &&
            Objects.equal(this.name, that.name) &&
            Objects.equal(this.url, that.url);
    }

    @Override
    public String toString()
    {
        return "Capability{" +
            "type='" + type + '\'' +
            ", name='" + name + '\'' +
            ", url='" + url + '\'' +
            '}';
    }
}
