package com.atlassian.plugins.navlink.producer.capabilities;

import com.atlassian.plugins.capabilities.api.AppWithCapabilities;
import com.google.common.base.Objects;
import com.google.common.collect.ImmutableMap;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;

import java.util.Map;

@Immutable
public class ApplicationWithCapabilities implements AppWithCapabilities
{
    protected static final DateTime NULL_DATE = new DateTime(0, DateTimeZone.UTC);

    protected final String id;
    protected final String name;
    protected final String type;
    protected final DateTime buildDate;
    protected final ImmutableMap<String, String> capabilities;

    public ApplicationWithCapabilities(@Nullable String type, @Nullable DateTime buildDate, Map<String, String> capabilities)
    {
        this.id = null;
        this.type = type;
        this.name = null;
        this.buildDate = toNonnull(buildDate);
        this.capabilities = ImmutableMap.copyOf(capabilities);
    }

    private DateTime toNonnull(DateTime buildDate)
    {
        return buildDate != null ? buildDate : NULL_DATE;
    }


    @Deprecated
    public String getId()
    {
        return id;
    }

    @Deprecated
    public String getName()
    {
        return name;
    }

    public String getType()
    {
        return type;
    }

    /**
     * Build date of the application, or a 'null' date (01/01/1970 UTC) if unknown.
     *
     * @return build date of the application
     */
    public DateTime getBuildDate()
    {
        return buildDate;
    }

    public boolean hasCapabilities()
    {
        return !capabilities.isEmpty();
    }

    /**
     * @deprecated use hasCapability(String key)
     * @param key
     * @return
     */
    public boolean hasCapability(CapabilityKey key)
    {
        return capabilities.containsKey(key.getKey());
    }

    /**
     * @deprecated use getCapabilityUrl(String key)
     * @param key
     * @return
     */
    @Nullable
    public String getCapabilityUrl(CapabilityKey key)
    {
        return getCapabilityUrl(key.getKey());
    }

    @Nonnull
    public ImmutableMap<String, String> getCapabilities()
    {
        return capabilities;
    }

    @Override
    public int hashCode()
    {
        return Objects.hashCode(id, name, type, buildDate);
    }

    @Override
    public boolean equals(final Object obj)
    {
        if (obj == this)
        {
            return true;
        }
        else if (!(obj instanceof ApplicationWithCapabilities))
        {
            return false;
        }
        else
        {
            final ApplicationWithCapabilities that = (ApplicationWithCapabilities) obj;
            return Objects.equal(type, that.type) &&
                    Objects.equal(name, that.name) &&
                    Objects.equal(id, that.id) &&
                    Objects.equal(buildDate, that.buildDate);
        }
    }

    @Override
    public String toString()
    {
        return "ApplicationWithCapabilities{" +
                "id='" + id + '\'' +
                ", name='" + name + '\'' +
                ", type='" + type + '\'' +
                ", buildDate='" + buildDate + '\'' +
                ", capabilities=" + capabilities +
                '}';
    }

    @Override
    public boolean hasCapability(String key)
    {
        return capabilities.containsKey(key);
    }

    @Override
    @Nullable
    public String getCapabilityUrl(String key)
    {
        return capabilities.get(key);
    }

}
