package com.atlassian.plugins.navlink.consumer.menu.services;

import com.atlassian.plugins.navlink.producer.navigation.NavigationLink;
import org.joda.time.DateTimeComparator;

import java.util.Comparator;

/**
 * <p/>
 * A comparator that expresses canonical order of the navigation links.
 *
 * <p/>
 * The following algorithm is applied to determine a consistent and meaningful ordering of navigation links:
 * <ul>
 *     <li>if weights of the links differ, use weights to determine the order (lower weights first)</li>
 *     <li>otherwise examine the <tt>applicationType</tt> attribute of the links:</li>
 *      <li>if links are of the same <tt>applicationType</tt> (i.e. we are comparing 2 applications of the same
 *         flavour), use the following attributes to determine the order:</li>
 *     <ul>
 *         <li>label - lexicographically ascending</li>
 *         <li>href - lexicographically ascending</li>
 *     </ul>
 *     <li>otherwise (i.e. links come from 2 applications of different type), use the following attributes to determine the order:</li>
 *      <ul>
 *         <li>build date - descending - 'younger' applications go first</li>
 *         <li>label - lexicographically ascending</li>
 *         <li>href - lexicographically ascending</li>
 *     </ul>
 * </ul>
 *
 * @see <a href="https://extranet.atlassian.com/x/aLXweg">Navigation Weight Conflict Resolution</a>
 */
public class NavigationLinkComparator implements Comparator<NavigationLink>
{
    public static final NavigationLinkComparator INSTANCE = new NavigationLinkComparator();

    /**
     * Enumeration of 'special' weight values.
     *
     */
    public static enum Weights
    {
        /**
         * Represents unknown weight (e.g. because the remote API does not support weights yet).
         *
         */
        UNSPECIFIED(Integer.MAX_VALUE),

        /**
         * The 'heaviest' weight that is treated as actual weight.
         *
         */
        MAX(Integer.MAX_VALUE-1);


        private final int value;

        private Weights(int value)
        {
            this.value = value;
        }

        public int value()
        {
            return value;
        }
    }

    @Override
    public int compare(NavigationLink first, NavigationLink second)
    {
        if (first == second) return 0;

        // TODO ROTP-783: remove this code once all products support weights API
        if (weightsNotYetSupported(first, second))
        {
            return LegacyNavLinkComparator.INSTANCE.compare(first, second);
        }

        final int comparison = compareInts(first.weight(), second.weight());
        if (comparison == 0)
        {
            if (sameApplicationType(first, second))
            {
                return compareSameApps(first, second);
            }
            else
            {
                return compareDifferentApps(first, second);
            }
        }
        else
        {
            return comparison;
        }
    }

    private boolean weightsNotYetSupported(NavigationLink one, NavigationLink two)
    {
        return one.weight() == Weights.UNSPECIFIED.value() || two.weight() == Weights.UNSPECIFIED.value();
    }

    private int compareInts(int one, int two)
    {
        // from Integer.compareTo()
        return (one < two ? -1 : (one == two ? 0 : 1));
    }

    private boolean sameApplicationType(NavigationLink first, NavigationLink second)
    {
        return first.getApplicationType().equals(second.getApplicationType());
    }

    /**
     * For same application type, don't bother using build date, just use label and href.
     *
     * @param first first navlink
     * @param second second navlink
     * @return the usual Comparator thing (fields as specified above)
     */
    private int compareSameApps(NavigationLink first, NavigationLink second)
    {
        int comparison = compareStrings(first.getLabel(), second.getLabel());
        if (comparison == 0) comparison = compareStrings(first.getHref(), second.getHref());
        return comparison;
    }

    /**
        * For different application type, build date first (later date first), then label, then href.
        *
        * @param first first navlink
        * @param second second navlink
        * @return the usual Comparator thing (fields as specified above)
        */
    private int compareDifferentApps(NavigationLink first, NavigationLink second)
    {
        int comparison = -DateTimeComparator.getInstance().compare(first.getBuildDate(), second.getBuildDate());
        if (comparison == 0) comparison = compareStrings(first.getLabel(), second.getLabel());
        if (comparison == 0) comparison = compareStrings(first.getHref(), second.getHref());
        return comparison;
    }

    private int compareStrings(String string1, String string2)
    {
        return String.CASE_INSENSITIVE_ORDER.compare(string1, string2);
    }
}
