package com.atlassian.plugins.navlink.consumer.menu.services;

import com.atlassian.plugins.capabilities.api.LinkedApplicationCapabilities;
import com.atlassian.failurecache.Cache;
import com.atlassian.failurecache.CacheFactory;
import com.atlassian.failurecache.Cacheable;
import com.atlassian.failurecache.Refreshable;
import com.atlassian.plugins.navlink.producer.capabilities.CapabilityKey;
import com.atlassian.plugins.navlink.producer.capabilities.RemoteApplicationWithCapabilities;
import com.atlassian.plugins.navlink.util.executor.DaemonExecutorService;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableSet;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;

import javax.annotation.Nullable;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Iterables.filter;

/**
 * Knows about the capabilities of linked applications. Serves all requests directly from its cache but still tries to
 * be up to date.
 * @since 3.2
 */
public class CachingLinkedApplicationCapabilitiesImpl implements Cacheable, InitializingBean, Runnable, Refreshable, RemoteApplications, LinkedApplicationCapabilities
{
    private static final long INITIAL_DELAY_IN_SECONDS = Long.getLong("navlink.capabilitiescache.initialdelay", 30);
    private static final long DELAY_IN_SECONDS = Long.getLong("navlink.capabilitiescache.delay", 10);
    private static final Logger logger = LoggerFactory.getLogger(CachingLinkedApplicationCapabilitiesImpl.class);

    private final Cache<RemoteApplicationWithCapabilities> cache;
    private final DaemonExecutorService executorService;

    @SuppressWarnings("UnusedDeclaration")
    public CachingLinkedApplicationCapabilitiesImpl(final DaemonExecutorService executorService, final CapabilitiesCacheLoader capabilitiesCacheLoader, final CacheFactory cacheFactory)
    {
        this(executorService, cacheFactory.createExpirationDateBasedCache(capabilitiesCacheLoader));
    }

    @VisibleForTesting
    CachingLinkedApplicationCapabilitiesImpl(final DaemonExecutorService executorService, final Cache<RemoteApplicationWithCapabilities> cache)
    {
        this.executorService = executorService;
        this.cache = cache;
    }

    @Override
    public Set<RemoteApplicationWithCapabilities> capableOf(final CapabilityKey capabilityKey)
    {
        return capableOf(capabilityKey.getKey());
    }

    @Override
    public Set<RemoteApplicationWithCapabilities> capableOf(final String capabilityKey)
    {
        checkNotNull(capabilityKey, "capabilityKey");
        return ImmutableSet.copyOf(filter(cache.getValues(), filterBy(capabilityKey)));
    }

    @Override
    public void run()
    {
        refreshCache();
    }

    @Override
    public int getCachePriority()
    {
        return 500;
    }

    @Override
    public void clearCache()
    {
        cache.clear();
    }

    @Override
    public ListenableFuture<?> refreshCache()
    {
        try
        {
            return cache.refresh();
        }
        catch (RuntimeException e)
        {
            logger.debug("Failed to refresh linked application capabilities cache", e);
            return Futures.immediateFailedFuture(e);
        }
    }

    @Override
    public void afterPropertiesSet() throws Exception
    {
        executorService.scheduleWithFixedDelay(this, INITIAL_DELAY_IN_SECONDS, DELAY_IN_SECONDS, TimeUnit.SECONDS);
    }

    private Predicate<RemoteApplicationWithCapabilities> filterBy(final String capabilityKey)
    {
        return new Predicate<RemoteApplicationWithCapabilities>()
        {
            @Override
            public boolean apply(@Nullable final RemoteApplicationWithCapabilities application)
            {
                return application != null && application.hasCapability(capabilityKey);
            }
        };
    }
}
