package com.atlassian.plugins.navlink.consumer.menu.rest;

import com.atlassian.plugins.navlink.consumer.menu.services.MenuService;
import com.atlassian.plugins.navlink.producer.navigation.NavigationLink;
import com.atlassian.plugins.rest.common.security.AnonymousAllowed;
import com.atlassian.sal.api.message.LocaleResolver;
import com.atlassian.sal.api.user.UserManager;
import com.google.common.base.Function;
import com.google.common.collect.Lists;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;

import static com.google.common.collect.Iterables.transform;
import static javax.ws.rs.core.MediaType.APPLICATION_JSON;

@Path ("/")
@AnonymousAllowed
public class MenuResource
{
    private final MenuService menuService;
    private final LocaleResolver localeResolver;
    private final UserManager userManager;

    public MenuResource(final MenuService menuService, final LocaleResolver localeResolver, final UserManager userManager)
    {
        this.menuService = menuService;
        this.localeResolver = localeResolver;
        this.userManager = userManager;
    }

    @GET
    @Path("{key}")
    @Produces (APPLICATION_JSON)
    public Response getMenuByKey(@Context HttpServletRequest hsr, @PathParam("key") final String key)
    {
        final Iterable<NavigationLink> menuNavigationLinks = menuService.getMenuItems(key, userManager.getRemoteUsername(hsr), localeResolver.getLocale());
        return createMenuNavigationLinkResponse(menuNavigationLinks);
    }

    /**
     * Gets the menu items for display in the new AppSwitcher.
     * Returns the same information as a calls to getMenuByKey for "home" and "custom-apps" keys, but with an order defined in the menuService across both sets.
     */
    @GET
    @Path("appswitcher")
    @Produces (APPLICATION_JSON)
    public Response getAppSwitcherMenu(@Context HttpServletRequest hsr)
    {
        final Iterable<NavigationLink> menuNavigationLinks = menuService.getAppSwitcherItems(userManager.getRemoteUsername(hsr));
        return createMenuNavigationLinkResponse(menuNavigationLinks);
    }


    private Response createMenuNavigationLinkResponse(Iterable<NavigationLink> menuNavigationLinks)
    {
        final Iterable<MenuNavigationLinkEntity> menuEntities = transform(menuNavigationLinks, convertToEntities());
        return Response.ok(menuEntities).build();
    }

    @Nonnull
    private Function<NavigationLink, MenuNavigationLinkEntity> convertToEntities()
    {
        return new Function<NavigationLink, MenuNavigationLinkEntity>()
        {
            @Override
            public MenuNavigationLinkEntity apply(@Nullable final NavigationLink menuNavigationLink)
            {
                return menuNavigationLink != null ? new MenuNavigationLinkEntity(menuNavigationLink) : null;
            }
        };
    }

}
