package com.atlassian.plugins.navlink.consumer.menu.rest;

import com.atlassian.plugins.navlink.producer.navigation.NavigationLink;
import com.atlassian.plugins.navlink.producer.navigation.links.SourceType;
import com.atlassian.plugins.navlink.producer.navigation.rest.MenuItemKey;
import com.google.common.base.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;

@XmlRootElement(name = "key")
@XmlAccessorType(XmlAccessType.PROPERTY)
public class MenuNavigationLinkEntity
{
    private MenuItemKey key;
    private String link;
    private String label;
    private String description;
    private String tooltip;
    private String iconUrl;
    private boolean local;
    private boolean self;

    public MenuNavigationLinkEntity()
    {
        // required by JAXB
    }

    public MenuNavigationLinkEntity(NavigationLink navLink)
    {
        this.key = retrieveKey(navLink);
        this.link = navLink.getHref();
        this.label = navLink.getLabel();
        this.tooltip = navLink.getTooltip();
        this.iconUrl = navLink.getIconUrl();
        this.local = navLink.getSource().type() == SourceType.LOCAL;
        this.self = navLink.isSelf();
    }


    @Nullable
    private static MenuItemKey retrieveKey(@Nonnull final NavigationLink navLink)
    {
        final String key = navLink.getKey();
        return key != null ? new MenuItemKey(key) : null;
    }

    public MenuItemKey getKey()
    {
        return key;
    }

    public String getLink()
    {
        return link;
    }

    public String getLabel()
    {
        return label;
    }

    public String getIconUrl()
    {
        return iconUrl;
    }

    public String getDescription()
    {
        return description;
    }

    public String getTooltip()
    {
        return tooltip;
    }

    public void setLink(String link)
    {
        this.link = link;
    }

    public void setLabel(String label)
    {
        this.label = label;
    }

    public void setIconUrl(String iconUrl)
    {
        this.iconUrl = iconUrl;
    }

    public void setTooltip(String tooltip)
    {
        this.tooltip = tooltip;
    }

    public void setDescription(String description)
    {
        this.description = description;
    }

    public void setKey(MenuItemKey key)
    {
        this.key = key;
    }


    /**
     * Indicates that this link was sourced locally (not necessarily pointing to a local location).  For example custom-apps may
     * be configured locally so will isLocal==true. Even though they point to a remote location e.g. www.google.com
     */
    public boolean isLocal()
    {
        return local;
    }


    /**
     * Indicates that the link points to the current host application.
     *
     * @return <code>true</code>, if the link points to the currently running application
     */
    public boolean isSelf()
    {
        return self;
    }

    public void setLocal(boolean local) {
        this.local = local;
    }

    @Override
    public boolean equals(Object o)
    {
        if (this == o)
        {
            return true;
        }
        if (o == null || getClass() != o.getClass())
        {
            return false;
        }

        final MenuNavigationLinkEntity that = (MenuNavigationLinkEntity) o;
        return Objects.equal(key, that.key) &&
                Objects.equal(link, that.link) &&
                Objects.equal(label, that.label) &&
                Objects.equal(description, that.description) &&
                Objects.equal(tooltip, that.tooltip) &&
                Objects.equal(iconUrl, that.iconUrl) &&
                Objects.equal(local, that.local) &&
                Objects.equal(self, that.self) &&
                Objects.equal(local, that.local);
    }

    @Override
    public int hashCode()
    {
        return Objects.hashCode(key, link, label, description, tooltip, iconUrl, local, self);
    }

    @Override
    public String toString()
    {
        return "MenuNavigationLinkEntity{" +
                ", key=" + key +
                ", link='" + link + '\'' +
                ", label='" + label + '\'' +
                ", description='" + description + '\'' +
                ", tooltip='" + tooltip + '\'' +
                ", iconUrl='" + iconUrl + '\'' +
                ", local=" + local +
                ", self=" + self +
                '}';
    }
}
