package com.atlassian.plugins.navlink.consumer.http.caching;

import com.atlassian.failurecache.ExpiringValue;
import com.atlassian.failurecache.util.date.Clock;
import com.atlassian.plugins.navlink.consumer.http.HeaderSearcher;
import org.apache.http.HttpResponse;

import javax.annotation.Nullable;
import javax.ws.rs.core.HttpHeaders;
import java.util.concurrent.TimeUnit;

import static com.google.common.base.Objects.firstNonNull;

class HttpCacheExpiryServiceImpl implements HttpCacheExpiryService
{
    private final Clock clock;

    public HttpCacheExpiryServiceImpl(final Clock clock)
    {
        this.clock = clock;
    }

    @Override
    public <V> ExpiringValue<V> createExpiringValueFrom(final HttpResponse response, @Nullable final V value)
    {
        final HeaderSearcher headerSearcher = new HeaderSearcher(response);
        final long responseDateInMillis = findResponseDateInMillis(headerSearcher, clock.getCurrentDate().getTime());
        final long maxAgeInMillis = findCacheControlMaxAgeInMillis(headerSearcher, 0L);
        final long staleInMillis = findCacheControlStaleInMillis(headerSearcher, 0L);

        final long softLimitInMillis = responseDateInMillis + maxAgeInMillis;
        final long hardLimitInMillis = softLimitInMillis + staleInMillis;
        return new ExpiringValue<V>(value, softLimitInMillis, hardLimitInMillis);
    }

    private long findResponseDateInMillis(final HeaderSearcher headerSearcher, final long defaultValue)
    {
        final Long dateHeaderInMillis = headerSearcher.findFirstHeaderValueAsDateInMillis(HttpHeaders.DATE);
        return firstNonNull(dateHeaderInMillis, defaultValue);
    }

    private long findCacheControlMaxAgeInMillis(final HeaderSearcher headerSearcher, final long defaultValue)
    {
        final Long maxAgeInSeconds = headerSearcher.findFirstHeaderElementAsLong(HttpHeaders.CACHE_CONTROL, "max-age");
        return maxAgeInSeconds != null ? TimeUnit.SECONDS.toMillis(maxAgeInSeconds) : defaultValue;
    }

    private long findCacheControlStaleInMillis(final HeaderSearcher headerSearcher, final long defaultValue)
    {
        final Long staleInSeconds = headerSearcher.findFirstHeaderElementAsLong(HttpHeaders.CACHE_CONTROL, "stale-while-revalidate");
        return staleInSeconds != null ? TimeUnit.SECONDS.toMillis(staleInSeconds) : defaultValue;
    }

}
