package com.atlassian.plugins.navlink.consumer.http;

import com.atlassian.failurecache.Cacheable;
import com.atlassian.failurecache.ExpiringValue;
import com.atlassian.plugins.navlink.consumer.http.caching.HttpCacheExpiryService;
import com.atlassian.util.concurrent.ResettableLazyReference;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpGet;
import org.springframework.beans.factory.DisposableBean;

import java.io.IOException;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;

import static com.google.common.base.Preconditions.checkNotNull;

public class HttpRequestFactoryImpl implements HttpRequestFactory, Cacheable, DisposableBean
{
    private final HttpClientFactory clientFactory;
    private final UserAgentProperty userAgentProperty;
    private final HttpCacheExpiryService httpCacheExpiryService;

    private final ResettableLazyReference<HttpClient> client = new ResettableLazyReference<HttpClient>()
    {
        @Override
        protected HttpClient create() throws Exception
        {
            return clientFactory.createHttpClient();
        }
    };

    public HttpRequestFactoryImpl(final HttpClientFactory clientFactory, final UserAgentProperty userAgentProperty, final HttpCacheExpiryService httpCacheExpiryService)
    {
        this.clientFactory = checkNotNull(clientFactory);
        this.userAgentProperty = checkNotNull(userAgentProperty);
        this.httpCacheExpiryService = checkNotNull(httpCacheExpiryService);
    }

    @Override
    public int getCachePriority()
    {
        return 100;
    }

    @Override
    public void clearCache()
    {
        if (client.isInitialized())
        {
            client.get().getConnectionManager().shutdown();
        }
        client.reset();
    }

    @Override
    public void destroy() throws Exception
    {
        clearCache();
    }

    @Override
    public <T> ExpiringValue<T> executeGetRequest(final String url, final ResponseHandler<T> responseHandler) throws IOException
    {
        return createRequest(checkNotNull(url)).executeRequest(new ExpiringValueResponseHandler<T>(checkNotNull(responseHandler), httpCacheExpiryService));
    }

    private HttpRequest createRequest(final String url)
    {
        return new HttpRequest(client.get(), createGetRequest(url));
    }

    private HttpGet createGetRequest(final String url)
    {
        final HttpGet request = new HttpGet(url);
        request.setHeader(HttpHeaders.ACCEPT, MediaType.APPLICATION_JSON);
        request.setHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON);
        request.setHeader(HttpHeaders.USER_AGENT, userAgentProperty.get());
        return request;
    }

}
