package com.atlassian.plugins.custom_apps;

import com.atlassian.applinks.api.ApplicationId;
import com.atlassian.applinks.api.ApplicationLink;
import com.atlassian.plugins.custom_apps.api.CustomApp;
import com.atlassian.plugins.custom_apps.api.CustomAppNotFoundException;
import com.atlassian.plugins.custom_apps.api.CustomAppService;
import com.atlassian.plugins.custom_apps.api.CustomAppsValidationException;
import com.atlassian.plugins.custom_apps.rest.data.validation.UrlFieldValidator;
import com.atlassian.plugins.navlink.consumer.CachingApplicationLinkService;
import com.atlassian.plugins.navlink.consumer.menu.services.NavigationLinkComparator;
import com.atlassian.plugins.navlink.consumer.menu.services.RemoteNavigationLinkService;
import com.atlassian.plugins.navlink.producer.navigation.NavigationLink;
import com.atlassian.plugins.navlink.producer.navigation.NavigationLinkPredicates;
import com.atlassian.plugins.navlink.producer.navigation.services.LocalNavigationLinkService;
import com.atlassian.sal.api.message.I18nResolver;
import com.atlassian.sal.api.message.LocaleResolver;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import static com.atlassian.plugins.navlink.producer.navigation.NavigationLinkPredicates.matchesCustomApp;
import static com.google.common.base.Predicates.or;

public class DefaultCustomAppService implements CustomAppService
{
    private static final Logger log = LoggerFactory.getLogger(DefaultCustomAppService.class);

    private final I18nResolver i18nResolver;
    private final CustomAppStore customAppStore;

    @Nonnull
    private final RemoteNavigationLinkService remoteNavigationLinkService;
    @Nonnull
    private final LocalNavigationLinkService localNavigationLinkService;
    @Nonnull
    private final CachingApplicationLinkService applicationLinkService;
    @Nonnull
    private final LocaleResolver localeResolver;

    public DefaultCustomAppService(
            @Nonnull final I18nResolver i18nResolver,
            @Nonnull final CustomAppStore customAppStore,
            @Nonnull final RemoteNavigationLinkService remoteNavigationLinkService,
            @Nonnull final LocalNavigationLinkService localNavigationLinkService,
            @Nonnull final CachingApplicationLinkService applicationLinkService,
            @Nonnull final LocaleResolver localeResolver
    )
    {
        this.i18nResolver = i18nResolver;
        this.customAppStore = customAppStore;
        this.remoteNavigationLinkService = remoteNavigationLinkService;
        this.localNavigationLinkService = localNavigationLinkService;
        this.applicationLinkService = applicationLinkService;
        this.localeResolver = localeResolver;
    }

    @Nonnull
    @Override
    public List<CustomApp> getCustomApps()
    {
        return Lists.newArrayList(Iterables.filter(customAppStore.getAll(), CustomAppPredicates.hasNoSourceApplicationUrl));
    }

    /**
     * Get all the custom apps and remote application (and remote custom apps) for display in the restful table used by
     * the admin page. This will also query the /home REST endpoint to refresh remote applications and custom apps.
     */
    @Override
    @Nonnull
    public List<CustomApp> getLocalCustomAppsAndRemoteLinks()
    {
        return getRefreshedLinks();
    }

    /**
     * gets a regenerated list of links as custom apps.
     * Combines remote and local links using the default ordering unless there is explicit local ordering.
     */
    private List<CustomApp> getRefreshedLinks()
    {
        Predicate<NavigationLink> allMenuItems = or(NavigationLinkPredicates.keyEquals("home"), NavigationLinkPredicates.keyEquals(NavigationLink.CUSTOM_APPS_KEY));
        Set<NavigationLink> remoteLinks = remoteNavigationLinkService.matching(localeResolver.getLocale(), allMenuItems);
        Set<NavigationLink> localLinks = localNavigationLinkService.matching(localeResolver.getLocale(), allMenuItems);

        List<NavigationLink> allLinks = new ArrayList<NavigationLink>();
        allLinks.addAll(remoteLinks);
        allLinks.addAll(localLinks);
        Collections.sort(allLinks, NavigationLinkComparator.INSTANCE);

        List<CustomApp> index = customAppStore.getAll();
        List<CustomApp> newList = new ArrayList<CustomApp>(index.size());

        List<CustomApp> indexedCustomApps = getLocallyIndexLinksAndCleanFromAllLinks(allLinks, index);
        newList.addAll(indexedCustomApps);

        List<CustomApp> newCustomApps = getUnindexedLinks(allLinks, maxId(newList) + 1);
        newList.addAll(newCustomApps);

        // if the user has never reordered the links, we need to reapply the default order in case it has changed
        if (!customAppStore.isCustomOrder())
        {
            Collections.sort(newList, CustomAppComparator.INSTANCE);
        }

        return newList;
    }

    /**
     * Anything not picked up by a the local index should be appended to the list of custom apps.
     */
    private List<CustomApp> getUnindexedLinks(List<NavigationLink> allLinks, int startIndex)
    {
        return Lists.newArrayList(Iterables.transform(allLinks, toUnindexed(startIndex)));
    }

    /**
     * If a local index has been defined it should be used to override the default ordering.
     */
    private List<CustomApp> getLocallyIndexLinksAndCleanFromAllLinks(List<NavigationLink> allLinks, List<CustomApp> current)
    {
        List<CustomApp> indexedLinks = Lists.newArrayList();

        for (CustomApp customApp : current)
        {
            NavigationLink navlink = findExactMatch(customApp, allLinks);
            if (navlink != null)
            {
                // Potentially need to correct custom apps entries prior to v3
                if(navlink.getSource().id() == null && navlink.getKey().equals(NavigationLink.CUSTOM_APPS_KEY))
                {
                    // if it is a local custom app it should always be editable.
                    indexedLinks.add(new CustomApp(customApp.getId(), navlink, null, null, customApp.getHide(),
                            customApp.getAllowedGroups(), true));
                }
                else if(navlink.getSource().id() == null && navlink.getKey().equals("home"))
                {
                    ApplicationLink sourceAppLink = getSourceAppLink(navlink);
                    // if its a local home entry we need to check/correct the Base Url
                    indexedLinks.add(new CustomApp(customApp.getId(), navlink,
                            resolveSourceApplicationUrl(sourceAppLink, navlink.getHref()),
                            resolveSourceApplicationName(sourceAppLink, navlink.getLabel()),
                            customApp.getHide(), customApp.getAllowedGroups(), customApp.getEditable()));
                }
                else if(navlink.getSource().id() != null)
                {
                    ApplicationLink sourceAppLink = getSourceAppLink(navlink);
                    //NB. if its a remote entry we need to check/correct the Base Url
                    indexedLinks.add(new CustomApp(customApp.getId(), navlink,
                            resolveSourceApplicationUrl(sourceAppLink, navlink.getHref()),
                            resolveSourceApplicationName(sourceAppLink, navlink.getLabel()),
                            customApp.getHide(), customApp.getAllowedGroups(), customApp.getEditable()));
                }
                else
                {
                    indexedLinks.add(new CustomApp(customApp.getId(), navlink, customApp.getSourceApplicationUrl(),
                            customApp.getSourceApplicationName(), customApp.getHide(), customApp.getAllowedGroups(), customApp.getEditable()));
                }
                allLinks.remove(navlink);
            }
        }
        return indexedLinks;
    }

    /**
     * Get the source ApplicationLink for a Navigationlink.
     */
    private ApplicationLink getSourceAppLink(final NavigationLink navLink)
    {
        if(navLink.getSource().id() == null)
        {
            // local navlinks do not have source application links.
            return null;
        }

        try
        {
            return applicationLinkService.getApplicationLink(new ApplicationId(navLink.getSource().id()));
        }
        catch(Exception e)
        {
            log.error("Unable to find source ApplicationLink  for '" + navLink + "'", e);
            return null;
        }
    }

    /**
     * Get the display URL from the source ApplicationLink, if this cannot be found use the default.
     */
    private String resolveSourceApplicationUrl(final ApplicationLink appLink, final String defaultUrl)
    {
        if (appLink == null)
        {
            return defaultUrl;
        }

        if (appLink.getDisplayUrl() == null)
        {
            return defaultUrl;
        }

        return appLink.getDisplayUrl().toASCIIString();
    }

    /**
     * Get the display Name from the source ApplicationLink, if this cannot be found use the default.
     */
    private String resolveSourceApplicationName(final ApplicationLink appLink, final String defaultName)
    {
        if (appLink == null)
        {
            return defaultName;
        }

        return appLink.getName();
    }

    private int maxId(List<CustomApp> apps)
    {
        int maxId = 0;
        for (CustomApp ca : apps)
        {
            int id = Integer.parseInt(ca.getId());
            if (id > maxId)
            {
                maxId = id;
            }
        }
        return maxId;
    }

    private NavigationLink findExactMatch(CustomApp c, List<NavigationLink> links)
    {
        return Iterables.find(links, matchesCustomApp(c), null);
    }

    @Override
    public CustomApp get(String id) throws CustomAppNotFoundException
    {
        List<CustomApp> apps = getRefreshedLinks();
        for (CustomApp app : apps)
        {
            if (app.getId().equals(id))
            {
                return app;
            }
        }
        throw createNotFoundException(id);
    }

    @Override
    public synchronized void delete(String id) throws CustomAppNotFoundException
    {
        List<CustomApp> apps = getRefreshedLinks();
        for (CustomApp app : apps)
        {
            if (app.getId().equals(id))
            {
                apps.remove(app);
                customAppStore.storeAll(apps);
                return;
            }
        }
        throw createNotFoundException(id);
    }

    @Override
    public synchronized CustomApp create(String displayName, String url, String baseUrl, boolean hide, List<String> newAllowedGroups)
            throws CustomAppsValidationException
    {
        displayName = checkField(CustomAppStore.DISPLAY_NAME, displayName);
        url = checkField(CustomAppStore.URL, url);
        List<CustomApp> apps = getRefreshedLinks();
        // created customApps should have a null base url and be editable
        CustomApp app = new CustomApp(nextId(apps), displayName, url, null, null, null, hide, newAllowedGroups, true);
        apps.add(app);
        customAppStore.storeAll(apps);
        return app;
    }

    @Override
    public synchronized CustomApp update(String id, String newDisplayName, String newUrl, boolean newHide, List<String> allowedGroups)
            throws CustomAppNotFoundException, CustomAppsValidationException
    {
        List<CustomApp> apps = getRefreshedLinks();
        for (int i = 0; i < apps.size(); ++i)
        {
            CustomApp app = apps.get(i);
            if (app.getId().equals(id))
            {
                newDisplayName = checkField(CustomAppStore.DISPLAY_NAME, newDisplayName);
                if (app.getSourceApplicationUrl() == null)
                { // only check the url for local custom apps
                    newUrl = checkField(CustomAppStore.URL, newUrl);
                }
                CustomApp updatedApp = new CustomApp(app.getId(), newDisplayName, newUrl, app.getSourceApplicationUrl(), app.getSourceApplicationName(), app.getSourceApplicationType(), newHide, allowedGroups, app.getEditable());
                apps.set(i, updatedApp);
                customAppStore.storeAll(apps);
                return updatedApp;
            }
        }
        throw createNotFoundException(id);
    }

    @Override
    public synchronized void moveAfter(int idToMove, int idToMoveAfter) throws CustomAppNotFoundException
    {
        List<CustomApp> apps = getLocalCustomAppsAndRemoteLinks();
        int indexToMove = findIndexById(apps, idToMove);
        int indexToMoveAfter = findIndexById(apps, idToMoveAfter);
        List<CustomApp> newList = new ArrayList<CustomApp>(apps.size());
        for (int i = 0; i < apps.size(); ++i)
        {
            if (i != indexToMove)
            {
                newList.add(apps.get(i));
                if (i == indexToMoveAfter)
                {
                    newList.add(apps.get(indexToMove));
                }
            }
        }
        customAppStore.storeAll(newList);
        customAppStore.setCustomOrder();
    }

    private int findIndexById(List<CustomApp> apps, int id) throws CustomAppNotFoundException
    {
        for (int i = 0; i < apps.size(); ++i)
        {
            if (id == Integer.parseInt(apps.get(i).getId()))
            {
                return i;
            }
        }
        throw createNotFoundException(Integer.toString(id));
    }

    @Override
    public synchronized void moveToStart(int idToMove) throws CustomAppNotFoundException
    {
        List<CustomApp> apps = getLocalCustomAppsAndRemoteLinks();
        int indexToMove = findIndexById(apps, idToMove);
        List<CustomApp> newList = new ArrayList<CustomApp>(apps.size());
        newList.add(apps.get(indexToMove));
        for (int i = 0; i < apps.size(); ++i)
        {
            if (i != indexToMove)
            {
                newList.add(apps.get(i));

            }
        }
        customAppStore.storeAll(newList);
        customAppStore.setCustomOrder();
    }

    /**
     * Validate a field value, returning a normalised value (usually the same as the input) or throwing a
     * CustomAppsValidationException with a message.
     */
    private String checkField(String fieldKey, String value) throws CustomAppsValidationException
    {
        if (StringUtils.isBlank(value))
        {
            throw new CustomAppsValidationException(fieldKey, i18nResolver.getText("must.not.be.empty"));
        }

        // URL field validation
        if (fieldKey.equals(CustomAppStore.URL))
        {
            value = fixUrl(value);
            if (!UrlFieldValidator.jira().isValid(value))
            {
                throw new CustomAppsValidationException(fieldKey, i18nResolver.getText("custom-apps.manage.validation.errors.url"));
            }
        }
        return value;
    }

    private String fixUrl(String url)
    {
        if (!url.startsWith("http://") && !url.startsWith("https://"))
        {
            return "http://" + url;
        }
        else
        {
            return url;
        }
    }

    private String nextId(List<CustomApp> apps)
    {
        int maxId = 0;
        for (CustomApp app : apps)
        {
            try
            {
                int id = Integer.parseInt(app.getId());
                if (id > maxId)
                {
                    maxId = id;
                }
            }
            catch (NumberFormatException nfe)
            {
                // do nothing -- this isn't a number and so can't conflict with the id we are about to generate
            }
        }
        return Integer.toString(maxId + 1);
    }

    private CustomAppNotFoundException createNotFoundException(String id)
    {
        return new CustomAppNotFoundException("No custom app found with id '" + id + "'");
    }

    private Function<NavigationLink, CustomApp> toUnindexed(final int startIndex)
    {
        return new ToUnindexed(startIndex);
    }

    private class ToUnindexed implements Function<NavigationLink, CustomApp>
    {
        private int startIndex = 0;
        private ToUnindexed(final int startIndex)
        {
            this.startIndex = startIndex;
        }

        @Override
        public CustomApp apply(NavigationLink navlink)
        {
            ApplicationLink sourceAppLink = getSourceAppLink(navlink);
            // all new items are by definition not editable, not hidden and not restricted by groups.
            boolean editable = false;
            boolean hidden = false;
            List<String> allowedGroups = Collections.emptyList();
            return new CustomApp(Integer.toString(startIndex++), navlink,
                    resolveSourceApplicationUrl(sourceAppLink, navlink.getHref()),
                    resolveSourceApplicationName(sourceAppLink, navlink.getLabel()),
                    hidden, allowedGroups, editable);
        }
    }
}
