package com.atlassian.paralyzer.core.reporting;

import com.atlassian.paralyzer.api.ExtendableType;
import com.atlassian.paralyzer.api.Extension;
import com.atlassian.paralyzer.api.TestResult;
import com.atlassian.paralyzer.api.TestSuite;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import lombok.var;
import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.io.OutputFormat;
import org.dom4j.io.XMLWriter;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Collection;
import java.util.Objects;
import java.util.Optional;

@Slf4j
@RequiredArgsConstructor
public class SurefireReport {
    private final File file;
    private final Collection<TestResult> results;
    private final TestSuite suite;
    @Setter
    private OutputFormat format = OutputFormat.createPrettyPrint();
    private int suiteFailures;
    private int suiteErrors;

    public SurefireReport(String filename, Collection<TestResult> results, TestSuite suite) {
        this.file = new File(filename);
        this.results = results;
        this.suite = suite;
    }

    public void writeReport() {
        log.info("Report is being written to file: " + file.getAbsolutePath());
        try (FileWriter fileWriter = new FileWriter(file)) {
            Document document = createXmlDocument();
            log.debug(document.asXML());
            XMLWriter writer = new XMLWriter(fileWriter, format);
            writer.write(document);
        } catch (IOException e) {
            throw new IllegalArgumentException(e);
        }
    }

    private Document createXmlDocument() {
        Document document = DocumentHelper.createDocument();
        Element testSuiteElement = createSuiteElement(document);

        suiteFailures = 0;
        suiteErrors = 0;

        for (var result : results) {
            createResultElement(testSuiteElement, result);
        }

        testSuiteElement
                .addAttribute("failures", Long.toString(suiteFailures))
                .addAttribute("errors", Long.toString(suiteErrors));
        return document;
    }

    private Element createSuiteElement(Document document) {
        String suiteName = getParamOfExtension(suite, "PrettyName", "name", String.class)
                .orElse(suite.getUniqueId());
        Double suiteTime = getParamOfExtension(suite, "ExecutionTime", "time", Double.class)
                .orElse(0.0);

        return document.addElement("testsuite")
                .addAttribute("name", suiteName)
                .addAttribute("tests", Integer.toString(results.size()))
                .addAttribute("time", Double.toString(suiteTime));
    }

    private <T> Optional<T> getParamOfExtension(ExtendableType extendable, String extensionType, String param, Class<T> aClass) {
        return extendable.getExtensions().stream()
                .filter(extension -> extension.getExtensionType().equals(extensionType))
                .map(extension -> (T) extension.getProperty(param))
                .filter(Objects::nonNull)
                .findAny();
    }

    private void createResultElement(Element testSuiteElement, TestResult result) {
        String className = getParamOfExtension(result, "DetailedInfo", "className", String.class)
                .orElse("");
        String methodName = getParamOfExtension(result, "DetailedInfo", "name", String.class)
                .orElse(result.getUniqueId());
        double time = getParamOfExtension(result, "DetailedInfo", "executionTime", Double.class)
                .orElse(0.0);

        Element testCaseElement = testSuiteElement.addElement("testcase")
                .addAttribute("classname", className)
                .addAttribute("name", methodName)
                .addAttribute("time", Double.toString(time));


        switch (result.getResult()) {
            case Ignored:
                testCaseElement.addElement("skipped");
                break;
            case Failed: {
                suiteFailures++;
                addFailureElement(result, testCaseElement, "failure");
                break;
            }
            case Error: {
                suiteErrors++;
                addFailureElement(result, testCaseElement, "error");
                break;
            }
        }
    }

    private void addFailureElement(TestResult result, Element testcase, String failure) {
        Extension failureDetails = result.getExtensions().stream().filter(extension -> extension.getExtensionType().equals("FailureDetails")).findAny().orElse(new Extension(""));
        String type = (String) failureDetails.getProperty("type");
        String message = (String) failureDetails.getProperty("message");
        String log = Optional.ofNullable(result.getLogs()).orElse("");

        testcase.addElement(failure)
                .addAttribute("type", type)
                .addAttribute("message", message)
                .addCDATA(log);
    }
}
