package com.atlassian.paralyzer.core.reporting;

import com.atlassian.paralyzer.api.Extension;
import com.atlassian.paralyzer.api.TestResult;
import com.atlassian.paralyzer.api.TestSuite;
import com.atlassian.paralyzer.api.engine.AfterSuite;
import com.atlassian.paralyzer.api.engine.BeforeSuite;
import com.atlassian.paralyzer.api.engine.TestEngineSettings;

import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Predicate;

/**
 * @since 1.0
 */
public class SuiteExecutionTimeListener implements BeforeSuite, AfterSuite {
    private final Map<String, Instant> startTimes = new ConcurrentHashMap<>();
    private final String extensionType = "ExecutionTime";
    private final String propertyName = "time";

    @Override
    public void afterSuite(TestSuite suite, List<TestResult> results, TestEngineSettings settings) {
        Instant now = Instant.now();
        Instant startTime = startTimes.get(getSuiteId(suite));
        if (startTime != null) {
            startTimes.remove(getSuiteId(suite));
            Double executionTime = (now.toEpochMilli() - startTime.toEpochMilli()) / 1000.0;
            suite.addExtension(new Extension(extensionType, Collections.singletonMap(propertyName, Optional.of(executionTime))));
        }
    }

    private String getSuiteId(TestSuite suite) {
        return suite.getTestEngineId() + "_" + suite.getUniqueId();
    }

    @Override
    public void beforeSuite(TestSuite suite, TestEngineSettings settings) {
        startTimes.put(getSuiteId(suite), Instant.now());
    }

    @Override
    public Predicate<String> getSupportedEnginePredicate() {
        return b -> true;
    }
}
