/**
 * The container for the drilldown view - ie. everything you see in the drilldown (header et al)
 * Controls the logic as to which individual notification entries should be collapsed
*/

MW.DrilldownAbstractView = Backbone.View.extend({

    className: "mw-notification-drilldown card",

    events: {
        "click .mw-drilldown-top" : "gotoMain",
        "click .mw-header .nav button": "gotoMain"
    },

    initialize: function(options) {
        this.globalEvents = options.globalEvents;
        this.reverseSlide = options.reverseSlide;

        this.globalEvents.bind("showNotificationDrilldownCard", this.showCard, this);
        this.globalEvents.bind("movetoDrilldownAbove", this.movetoDrilldownAbove, this);
        this.globalEvents.bind("movetoDrilldownBelow", this.movetoDrilldownBelow, this);
        this.globalEvents.bind("cleanupNestedCard", this.removeElementAndBindings, this);

        var kb = MW.KeyboardShortcuts.setupContext("notificationDrilldown-" + this.model.get("aggregateKey"),
            "showNotificationDrilldownCard", this);
        kb.addShortcut(13, this.openDrilldownObject); // enter
        kb.addShortcut(106, this.gotoAboveNotification); // j
        kb.addShortcut(107, this.gotoBelowNotification); // k
        kb.addShortcut(110, this.movetoDrilldownBelow); // n
        kb.addShortcut(112, this.movetoDrilldownAbove); // p
        kb.addShortcut(115, this.createTaskFromDrilldown); // s
        kb.addShortcut(117, this.gotoMain); // u
        this.kb = kb;
    },

    gotoAboveNotification: function() {
        this.globalEvents.trigger("highlightNotificationAbove");
        this.renderDrilldown(false);
    },

    gotoBelowNotification: function() {
        this.globalEvents.trigger("highlightNotificationBelow");
        this.renderDrilldown(true);
    },

    renderDrilldown: function(reverseSlide) {
        var focusedNotification = MW.Notifications.focused("mainFocused");
        if (this.model != focusedNotification) {
            this.globalEvents.trigger("renderDrilldown",focusedNotification, false, reverseSlide);
        }
    },

    createTaskFromDrilldown: function() {
        this.headerView.createTask();
    },

    movetoDrilldownAbove: function() {
        var oldFocused = this.model.focused();
        var focused = this.model.focusOlder();
        if (oldFocused === focused) {
            MW.App.scrollToDrilldownHeader();
        } else {
            MW.App.scrollToFocusedDrilldown();
            if (_(this.collapsedGroup).contains(focused)) {
                this.showOlder(function(){
                    // using kb shortcut - scroll to focused (WDAY-1531)
                    MW.App.scrollToFocusedDrilldown();
                });
            }
        }
    },

    movetoDrilldownBelow: function() {
        this.model.focusNewer();
        MW.App.scrollToFocusedDrilldown();
    },

    openDrilldownObject: function() {
        var urlToOpen = this.model.focused().get("url");
        window.open(urlToOpen);
    },

    render: function() {
        var $el = MW.$(this.el),
            that = this,
            notificationGroup = this.model,
            notifications = this.model.notifications;

        // Create the skeleton HTML where everything will go
        this.renderTemplate($el);

        var $headerDiv = $el.find(".mw-drilldown-header"),
            $entriesDiv = $el.find(".mw-drilldown-body");

        // To render the view, we need information about number of new notifications, so calculate it now
        this.collapsedGroup = _.filter(notifications, function(notification) {
            return notification.get("read");
        });
        var collapsedGroupCount = this.collapsedGroup.length;

        var numUpdates = notifications.length - collapsedGroupCount;

        this.headerView = new MW.HeaderView({
            el: $headerDiv[0],
            model: notificationGroup,
            numUpdates: numUpdates,
            globalEvents: this.globalEvents
        });

        this.headerView.render();
        this.entryViews = [];

        // Generate the notifications
        _.each(notifications, function(notification){
            var view = new MW.NotificationEntryView({model: notification, globalEvents: that.globalEvents,
                attributes: {
                    "notification-id": notification.get("id")
                }
            });
            $entriesDiv.prepend(view.render().el);
            that.entryViews.push(view);
        });

        // Handle collapsed state of notifications
        var $collapsedEntries = $el.find(".mw-drilldown-list.collapsed");
        // All read, so reduce count by one since we are going to display the most recent item
        if (collapsedGroupCount === notifications.length) {
            collapsedGroupCount -= 1;
            $collapsedEntries.last().removeClass("collapsed");
            // Re-define what the list of collapsed entries are
            $collapsedEntries = $el.find(".mw-drilldown-list.collapsed");
        }

        $collapsedEntries.wrapAll("<div class='collapsed-group'/>");

        // Only show header if there are collapsed entries
        if (collapsedGroupCount) {
            $entriesDiv.prepend("<div class='mw-collapsed-header'></div>");
            var $collapsedHeader = $entriesDiv.find(".mw-collapsed-header");

            $collapsedHeader.click(function(e) {
                // Weird click behaviour on mobile without this
                setTimeout(function(){
                    that.showOlder(function(){
                        that.model.focusOldest();
                    });
                }, 0);
                e.preventDefault();
            });
            var olderText = this.model.i18ns("older", collapsedGroupCount);
            $collapsedHeader.html("<a href='#'>" + olderText + "</a>");
        }

        return this;
    },

    showOlder: function(callback) {
        var $collapsedHeader = this.$el.find('.mw-collapsed-header');
        if ($collapsedHeader.length === 0) {
            return;
        }
        this.showCollapsedGroup(MW.$(".collapsed-group"), callback);
        $collapsedHeader.remove();
    },

    gotoMain: function(){
        this.globalEvents.trigger("showNotificationCard");
    },

    removeElementAndBindings: function(){
        if (!this.$el.hasClass('showing')) {
            this.globalEvents.off(null, null, this);
            this.kb.cleanup();
            this.headerView.removeBindings();
            _.each(this.entryViews, function(view) {
                view.cleanup();
            });
            this.undelegateEvents();
            this.remove();
        }
    },

    showCard: function(slideTime) {
        if (!this.$el.hasClass('showing')) {
            MW.Router.navigateToNotification(this.model.get("aggregateKey"));
            if (this.reverseSlide) {
                this.showCardAnimationReverseSlide(slideTime);
            } else {
                this.showCardAnimation(slideTime);
            }
        }
    }
});
