package com.atlassian.multitenant.impl.datastore;

import com.atlassian.multitenant.Tenant;
import com.atlassian.multitenant.impl.MultiTenantDatastore;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;

/**
 * Datastore that maintains a system tenant, and delegates to another datastore if not requesting the system tenant
 */
public class DelegatingSystemTenantDatastore implements MultiTenantDatastore
{
    private final MultiTenantDatastore delegate;
    private final Tenant systemTenant;

    public DelegatingSystemTenantDatastore(final MultiTenantDatastore delegate, final Tenant systemTenant)
    {
        this.delegate = delegate;
        this.systemTenant = systemTenant;
    }

    public Tenant get(final String name)
    {
        if (systemTenant.getName().equals(name))
        {
            return systemTenant;
        }
        return delegate.get(name);
    }

    public Collection<Tenant> getAll()
    {
        ArrayList<Tenant> results = new ArrayList<Tenant>();
        results.add(systemTenant);
        results.addAll(delegate.getAll());
        return results;
    }

    public Tenant getByHostname(final String hostname)
    {
        if (systemTenant.getHostnames().contains(hostname))
        {
            return systemTenant;
        }
        return delegate.getByHostname(hostname);
    }

    public void addTenant(final Tenant tenant)
    {
        if (systemTenant.getName().equals(tenant.getName()))
        {
            throw new IllegalArgumentException("Can not add tenant with same name as system tenant");
        }
        delegate.addTenant(tenant);
    }

    public void removeTenant(final Tenant tenant)
    {
        if (systemTenant.getName().equals(tenant.getName()))
        {
            throw new IllegalArgumentException("Can not remove system tenant");
        }
        delegate.removeTenant(tenant);
    }
}
