package com.atlassian.multitenant.impl;

import com.atlassian.multitenant.TenantReference;
import com.atlassian.multitenant.Tenant;

import java.util.LinkedList;

/**
 * Default implemntation of the TenantReference
 */
public class DefaultTenantReference implements TenantReference
{
    // Use a LinkedList as a stack
    private final ThreadLocal<LinkedList<Tenant>> CONTEXT = new ThreadLocal<LinkedList<Tenant>>() {
        protected LinkedList<Tenant> initialValue()
        {
            return new LinkedList<Tenant>();
        }
    };

    public Tenant get() throws IllegalStateException
    {
        LinkedList<Tenant> context = CONTEXT.get();
        if (context.isEmpty())
        {
            throw new IllegalStateException("Attempt to retrieve multi tenant context when no context is set");
        }
        return context.getFirst();
    }

    public boolean isSet()
    {
        LinkedList<Tenant> context = CONTEXT.get();
        return !context.isEmpty();
    }

    public void set(Tenant tenant, boolean allowOverride) throws IllegalStateException
    {
        LinkedList<Tenant> context = CONTEXT.get();
        if (!allowOverride && !CONTEXT.get().isEmpty())
        {
            // Something has gone wrong, we will throw an exception, something will die and it will be logged
            // somewhere so we can do something about it, but if we don't clean up, this thread will now throw this
            // exception on every call, so clean up first.
            context.removeFirst();
            throw new IllegalStateException("Attempt to set multi tenant context when one already exists");
        }
        context.addFirst(tenant);
    }

    public void remove() throws IllegalStateException
    {
        LinkedList<Tenant> context = CONTEXT.get();
        if (context.isEmpty())
        {
            throw new IllegalStateException("Attempt to remove multi tenant context when none was set");
        }
        context.removeFirst();
    }
}
