package com.atlassian.multitenant;

import java.lang.reflect.Method;
import java.util.Set;

/**
 * Factory for constructing components that are transparently aware of the current tenant.
 */
public interface MultiTenantComponentFactory
{
    /**
     * The MultiTenantComponentMap is an intrusive form of multi tenant support, requiring a call to get() before
     * each invocation of the underlying component.  This will return a builder for a map.
     *
     * @param creator The creation callback for creating a new component when required.
     * @return The map builder
     */
    <C> MultiTenantComponentMapBuilder<C> createComponentMapBuilder(MultiTenantCreator<C> creator);

    /**
     * The MultiTenantComponentMap is an intrusive form of multi tenant support, requiring a call to get() before
     * each invocation of the underlying component.  The components will be lazy loaded.
     *
     * @param creator The creation callback for creating a new component when required.
     * @return The map
     */
    <C> MultiTenantComponentMap<C> createComponentMap(MultiTenantCreator<C> creator);

    /**
     * Create a dynamic proxy that delegates to the instance returned by the component map for the current tenant.
     *
     * @param map The map containing the actual components
     * @param classLoader The classloader to create the proxy class in
     * @param interfaces The interfaces the proxy must implement
     * @return The component
     */
    <C> Object createComponent(MultiTenantComponentMap<C> map, ClassLoader classLoader, Class<? super C>... interfaces);

    /**
     * Create a dynamic proxy that delegates to the instance returned by the component map for the current tenant.
     *
     * @param map The map containing the actual components
     * @param inter The interface the proxy must implement
     * @return The component
     */
    <C> C createComponent(MultiTenantComponentMap<C> map, Class<C> inter);

    /**
     * Convenient method to create a dynamic proxy that uses the given creator.  Uses lazy loading.
     *
     * @param creator The creation callback for creating a new component when required.
     * @param inter The interface the proxy should implement
     * @return The component proxy
     */
    <C> C createComponent(MultiTenantCreator<C> creator, Class<C> inter);

    /**
     * Convenient method for creating a component when all that's needed to construct the new delegate components is a
     * simple call to the no arg constructor to the super class.
     *
     * @param clazz The implementing class
     * @param inter The interface the returned proxy should implement
     * @return The component proxy
     */
    <C> C createComponent(Class<? extends C> clazz, Class<C> inter);

    /**
     * Create a dynamic proxy that delegates to the instance returned by the component map for the current tenant.
     * Allows configuration of a set of methods that a call will invoke on all tenants.
     *
     * @param map The map that this proxy will be backed by
     * @param invokeForAllMethods The methods to invoke on all tenants
     * @param inter The interface to proxy
     * @return The component
     */
    <C> C createComponent(MultiTenantComponentMap<C> map, Set<Method> invokeForAllMethods, Class<C> inter);

    /**
     * Create a dynamic proxy that delegates to the instance returned by the component map for the current tenant.
     * Allows configuration of a set of methods that a call will invoke on all tenants.
     *
     * @param map The map that this proxy will be backed by
     * @param classLoader The classloader that the proxy will be loaded into
     * @param invokeForAllMethods The methods to invoke on all tenants
     * @param interfaces The interfaces to proxy
     * @return The component
     */
    <C> Object createComponent(MultiTenantComponentMap<C> map, ClassLoader classLoader, Set<Method> invokeForAllMethods,
            Class<? super C>... interfaces);

    /**
     * Create an enhanced component based on the given map.  The class must have a noarg constructor.
     *
     * @param map The map for the enhanced component to delegate to
     * @param superClass The superclass to enhance
     * @param <C> The type of component that is being enhanced
     * @return The enhanced component
     */
    <C> C createEnhancedComponent(MultiTenantComponentMap<C> map, Class superClass);

    /**
     * Convenience method for creating an enhanced component with the given creator.  If a standard lazily loaded map is
     * not what is required, create a custom map and pass to the appropriate method.
     *
     * @param creator The creator to create new instances of the class
     * @param superClass The class to enhance
     * @return The enhanced proxy
     */
    <C> C createEnhancedComponent(MultiTenantCreator<C> creator, Class<C> superClass);

    /**
     * Convenient method for creating an enhanced component when all that's needed to construct the new delegate
     * components is a simple call on the no arg constructor of the super class.
     *
     * @param superClass The superClass to enhance
     * @return The enhanced component
     */
    <C> C createEnhancedComponent(Class<C> superClass);
}
