package com.atlassian.maven.plugins.amps;

import static com.atlassian.maven.plugins.amps.util.PropertyUtils.parse;
import static com.google.common.base.MoreObjects.firstNonNull;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.trimToEmpty;
import static org.apache.commons.lang3.builder.ToStringStyle.SHORT_PREFIX_STYLE;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.springframework.jdbc.datasource.DriverManagerDataSource;
import org.springframework.jdbc.support.DatabaseMetaDataCallback;
import org.springframework.jdbc.support.JdbcUtils;
import org.springframework.jdbc.support.MetaDataAccessException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;

/**
 * Definition of a datasource. For more information about the properties, see
 * http://cargo.codehaus.org/DataSource+and+Resource+Support
 *
 * @since 3.11
 */
public class DataSource {

    private static final String[] FIELDS_TO_EXCLUDE_FROM_TO_STRING = {"password", "systemPassword"};

    private static final char PROPERTY_DELIMITER = ';';

    private static final char PROPERTY_KEY_VALUE_DELIMITER = '=';

    /** Connection url, such as "jdbc:h2:file:/path/to/database/file" */
    private String url;

    /** database schema, such as "public", "dbo" */
    private String schema;

    /** Driver, such as "org.h2.Driver" */
    private String driver;

    /** Username, e.g. "sa" */
    private String username;

    /** Password. May be empty. */
    private String password;

    /** JNDI name, such as jdbc/JiraDS */
    private String jndi;

    /** Type of driver: "java.sql.Driver" or "javax.sql.XADataSource". Will not be forwarded to Cargo if empty. */
    private String type;

    /** Which transaction support. LOCAL_TRANSACTION or XA_TRANSACTION. Will not be forwarded to Cargo if empty. */
    private String transactionSupport;

    /** Properties to pass to the driver. Semi-colon delimited string. Will not be forwarded to Cargo if empty. */
    private String properties;

    /**
     * Cargo-style string to pass to Cargo in the "cargo.datasource.datasource" property. If set, the other properties
     * will not be read.
     *
     * <p>Example: cargo.datasource.username=sa|cargo.datasource.password...|...
     */
    private String cargoString;

    /**
     * Additional libraries required in the container (e.g. Tomcat) to support the driver. Example with a random
     * library:
     *
     * <pre>{@code
     * <libArtifacts>
     *   <libArtifact>
     *      <groupId>postgres</groupId>
     *      <artifactId>postgres</artifactId>
     *      <version>9.1-901-1.jdbc4</artifactId>
     *   </libArtifact>
     * </libArtifacts>
     * }</pre>
     */
    private List<LibArtifact> libArtifacts = new ArrayList<>();

    /**
     * The JDBC URL of the system database, being the one to which a DBA would connect in order to drop or create a
     * schema, database, user, etc.
     *
     * <p>N.B. renaming this field to (for example) "systemUrl" would break field injection for existing AMPS
     * configurations
     */
    private String defaultDatabase;

    /** The username of a database user who can drop and create schemas, databases, users, etc. */
    private String systemUsername;

    /** The password of the user identified by the {@code systemUsername}. May be empty. */
    private String systemPassword;

    /**
     * The file from which to import data. If not null, the file to which this points should be in the correct format
     * for the configured {@link #importMethod}.
     */
    private String dumpFilePath;

    /**
     * How AMPS should import the dump file: sql : AMPS will use JDBC to import SQL dump file, file must contain
     * standard SQL psql: AMPS will use Postgres psql to import SQL dump file. Eg: psql -f jira_63_postgres91_dump.sql
     * -U jira_user jiradb impdp: AMPS will use Oracle impdp to import dump file. Eg: impdp jira_user/jira_pwd
     * directory=data_pump_dir DUMPFILE=<dumpFilePath> sqlcmd: AMPS will use SQL Server sqlcmd to restore backup file.
     * Eg: sqlcmd -s localhost -Q "RESTORE DATABASE JIRA FROM DISK='d:\jira_63_sqlserver_2008.bak'"
     */
    private String importMethod = "sql";

    /** Whether the prepare-database goal should drop and re-create the database and the product's user. */
    @SuppressWarnings("unused")
    private boolean dropAndReCreateDatabase;

    /**
     * An optional SQL-92 query for validating the database connection. Not all products use this.
     *
     * @since 8.3
     */
    private String validationQuery;

    /** Constructor. */
    public DataSource() {
        // Empty
    }

    /**
     * Indicates whether AMPS should drop and re-create the application database during the pre-integration-test phase.
     *
     * @since 8.3
     */
    public boolean isDropAndReCreateDatabase() {
        return dropAndReCreateDatabase;
    }

    public String getCargoString() {
        if (cargoString != null) {
            return cargoString;
        }

        final List<String> cargoProperties = new ArrayList<>();
        cargoProperties.add("cargo.datasource.url=" + firstNonNull(url, ""));
        cargoProperties.add("cargo.datasource.driver=" + firstNonNull(driver, ""));
        cargoProperties.add("cargo.datasource.username=" + firstNonNull(username, ""));
        cargoProperties.add("cargo.datasource.password=" + firstNonNull(password, ""));
        cargoProperties.add("cargo.datasource.jndi=" + firstNonNull(jndi, ""));
        if (!isBlank(type)) {
            cargoProperties.add("cargo.datasource.type=" + type);
        }
        if (!isBlank(transactionSupport)) {
            cargoProperties.add("cargo.datasource.transactionsupport=" + transactionSupport);
        }
        if (!isBlank(properties)) {
            cargoProperties.add("cargo.datasource.properties=" + properties);
        }

        cargoString = StringUtils.join(cargoProperties, "|");
        return cargoString;
    }

    /**
     * Fills in any missing fields of {@code this} datasource with the values of the given datasource.
     *
     * @param defaultValues the datasource from which to read the default values
     */
    public void copyMissingValuesFrom(final DataSource defaultValues) {
        if (this.jndi == null) this.jndi = defaultValues.jndi;
        if (this.url == null) this.url = defaultValues.url;
        if (this.schema == null) this.schema = defaultValues.schema;
        if (this.driver == null) this.driver = defaultValues.driver;
        if (this.username == null) this.username = defaultValues.username;
        if (this.password == null) this.password = defaultValues.password;
        if (this.type == null) this.type = defaultValues.type;
        if (this.transactionSupport == null) this.transactionSupport = defaultValues.transactionSupport;
        if (this.properties == null) this.properties = defaultValues.properties;
    }

    public String getUrl() {
        return url;
    }

    public void setUrl(String url) {
        this.url = url;
    }

    public String getSchema() {
        return schema;
    }

    public void setSchema(String schema) {
        this.schema = schema;
    }

    public String getDriver() {
        return driver;
    }

    public void setDriver(String driver) {
        this.driver = driver;
    }

    public String getUsername() {
        return username;
    }

    public void setUsername(String username) {
        this.username = username;
    }

    public String getPassword() {
        return password;
    }

    public void setPassword(String password) {
        this.password = password;
    }

    public String getJndi() {
        return jndi;
    }

    public void setJndi(String jndi) {
        this.jndi = jndi;
    }

    public String getType() {
        return type;
    }

    public void setType(String type) {
        this.type = type;
    }

    public String getTransactionSupport() {
        return transactionSupport;
    }

    public void setTransactionSupport(String transactionSupport) {
        this.transactionSupport = transactionSupport;
    }

    public String getProperties() {
        return properties;
    }

    public void setProperties(String properties) {
        this.properties = properties;
    }

    public List<LibArtifact> getLibArtifacts() {
        return libArtifacts;
    }

    public void setLibArtifacts(List<LibArtifact> libArtifacts) {
        this.libArtifacts = libArtifacts;
    }

    public void setCargoString(String cargoString) {
        this.cargoString = cargoString;
    }

    /**
     * Returns the URL of the default database, i.e. the one to which a DBA would connect to drop or create databases
     * and users.
     *
     * @return a JDBC URL
     */
    public String getSystemUrl() {
        return defaultDatabase;
    }

    /**
     * Sets the URL of the default database, i.e. the one to which a DBA would connect to drop or create databases and
     * users.
     *
     * @param jdbcUrl the JDBC URL of the default database
     */
    public void setSystemUrl(final String jdbcUrl) {
        this.defaultDatabase = jdbcUrl;
    }

    public String getSystemUsername() {
        return systemUsername;
    }

    public void setSystemUsername(String systemUsername) {
        this.systemUsername = systemUsername;
    }

    public String getSystemPassword() {
        return systemPassword;
    }

    public void setSystemPassword(String systemPassword) {
        this.systemPassword = systemPassword;
    }

    public String getDumpFilePath() {
        return dumpFilePath;
    }

    public void setDumpFilePath(String dumpFilePath) {
        this.dumpFilePath = dumpFilePath;
    }

    public String getImportMethod() {
        return importMethod;
    }

    public void setImportMethod(String importMethod) {
        this.importMethod = importMethod;
    }

    @Override
    public String toString() {
        // Used in error messages, etc.
        return new ReflectionToStringBuilder(this, SHORT_PREFIX_STYLE)
                .setExcludeFieldNames(FIELDS_TO_EXCLUDE_FROM_TO_STRING)
                .toString();
    }

    /**
     * Returns the JDBC data source for this instance, connecting as the configured system user and applying any
     * configured driver properties.
     *
     * @return see above
     */
    public javax.sql.DataSource getJdbcDataSource() {
        final DriverManagerDataSource dataSource =
                new DriverManagerDataSource(defaultDatabase, systemUsername, systemPassword);
        dataSource.setConnectionProperties(parse(properties, PROPERTY_KEY_VALUE_DELIMITER, PROPERTY_DELIMITER));
        return dataSource;
    }

    /**
     * Queries this data source for its JDBC metadata and applies the given callback.
     *
     * @param callback the callback to apply
     * @return the non-null return value of the callback, or none if null or there was an error
     */
    @Nonnull
    public <T> Optional<T> getJdbcMetaData(@Nonnull final DatabaseMetaDataCallback callback) {
        try {
            @SuppressWarnings("unchecked")
            final T metaData = (T) JdbcUtils.extractDatabaseMetaData(getJdbcDataSource(), callback);
            return Optional.ofNullable(metaData);
        } catch (final ClassCastException | MetaDataAccessException e) {
            return Optional.empty();
        }
    }

    /**
     * Adds the given JDBC driver property.
     *
     * @param name the property name
     * @param value the property value
     */
    public void addProperty(final String name, final String value) {
        final String newProperty = name + PROPERTY_KEY_VALUE_DELIMITER + value;
        if (isBlank(properties)) {
            properties = newProperty;
        } else {
            properties += PROPERTY_DELIMITER + newProperty;
        }
    }

    /**
     * Returns the JDBC driver properties in the <a
     * href="http://www.mojohaus.org/sql-maven-plugin/execute-mojo.html#driverProperties">format required by the <code>
     * sql-maven-plugin</code></a>.
     *
     * @return see above
     */
    public String getSqlPluginJdbcDriverProperties() {
        return trimToEmpty(properties).replace(PROPERTY_DELIMITER, ',');
    }

    /**
     * Returns the SQL query that validates the database connection. Not all products use this.
     *
     * @return an SQL-92 query, or {@code null} if none is provided
     * @since 8.3
     */
    @Nullable public String getValidationQuery() {
        return validationQuery;
    }

    /**
     * Sets the SQL query that validates the database connection. Not all products use this.
     *
     * @param validationQuery the SQL-92 query to set
     * @since 8.3
     */
    public void setValidationQuery(final String validationQuery) {
        this.validationQuery = validationQuery;
    }
}
