package com.atlassian.maven.plugins.amps;

import com.atlassian.maven.plugins.amps.product.ImportMethod;
import com.atlassian.maven.plugins.amps.product.jira.JiraDatabase;
import com.atlassian.maven.plugins.amps.product.jira.JiraDatabaseFactory;
import com.atlassian.maven.plugins.amps.product.jira.JiraDatabaseType;
import org.apache.maven.model.Dependency;
import org.apache.maven.model.Plugin;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.codehaus.plexus.util.xml.Xpp3Dom;

import javax.annotation.Nullable;
import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import static com.atlassian.maven.plugins.amps.product.ImportMethod.SQL;
import static com.atlassian.maven.plugins.amps.product.ProductHandlerFactory.JIRA;
import static com.atlassian.maven.plugins.amps.product.jira.JiraDatabaseType.getDatabaseType;
import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isNotEmpty;
import static org.apache.maven.plugins.annotations.ResolutionScope.TEST;
import static org.twdata.maven.mojoexecutor.MojoExecutor.goal;

/**
 * For each Jira instance to be run that has exactly one {@link DataSource} configured:
 * <ol>
 *     <li>Drops and re-creates the database, then</li>
 *     <li>Imports the configured data file, using either direct SQL or a db-specific import tool</li>
 * </ol>
 */
@Mojo(name = "prepare-database", requiresDependencyResolution = TEST)
public class PrepareDatabaseMojo extends AbstractTestGroupsHandlerMojo {
    @Parameter(property = "maven.test.skip", defaultValue = "false")
    private boolean testsSkip;

    @Parameter(property = "skipTests", defaultValue = "false")
    private boolean skipTests;

    @Parameter(property = "db.dump.file.path")
    private String dumpFilePath;

    @Parameter(property = "import.method")
    private String importMethod;

    @Parameter(property = "db.default.database")
    private String defaultDatabase;

    @Parameter(property = "db.system.username")
    private String systemUsername;

    @Parameter(property = "db.system.password")
    private String systemPassword;

    @Override
    protected void doExecute() throws MojoExecutionException {
        if (testsSkip || skipTests) {
            getLog().info("Pre integration tests skipped");
            return;
        }
        for (final ProductExecution jira : getJiraExecutions()) {
            final DataSource dataSource = getOnlyDataSource(jira);
            if (dataSource != null) {
                prepareDatabase(jira, dataSource);
            }
        }
    }

    private Collection<ProductExecution> getJiraExecutions() throws MojoExecutionException {
        return getProductExecutions().stream()
                .filter(execution -> JIRA.equals(execution.getProduct().getId()))
                .collect(toList());
    }

    @Nullable
    private DataSource getOnlyDataSource(final ProductExecution jiraExecution) {
        final List<DataSource> dataSources = jiraExecution.getProduct().getDataSources();
        switch (dataSources.size()) {
            case 0:
                getLog().info("No dataSource configured for pre-integration-test");
                return null;
            case 1:
                return dataSources.get(0);
            default:
                getLog().info("Multiple dataSources not supported. Configuration has these " +
                        dataSources.size() + " dataSources:");
                for (DataSource dataSource : dataSources) {
                    getLog().info("- Database URL: " + dataSource.getUrl());
                }
                return null;
        }
    }

    private void prepareDatabase(final ProductExecution jiraExecution, final DataSource dataSource)
            throws MojoExecutionException {
        final JiraDatabaseType databaseType = getDatabaseType(dataSource)
                .orElseThrow(() -> new MojoExecutionException(
                        "Could not detect database type for dataSource: " + dataSource));
        dataSource.getLibArtifacts().addAll(getJdbcDriverArtifacts(jiraExecution, databaseType));
        populateParameters(dataSource);
        final JiraDatabase jiraDatabase = new JiraDatabaseFactory(getLog()).getJiraDatabase(dataSource)
                .orElseThrow(() -> new MojoExecutionException("Cannot get database for " + dataSource));
        dropAndCreate(jiraDatabase);
        importDumpFile(jiraDatabase);
    }

    private Collection<LibArtifact> getJdbcDriverArtifacts(
            final ProductExecution jiraExecution, final JiraDatabaseType databaseType)
            throws MojoExecutionException {
        // Brute force approach: add all of the product's lib artifacts, and hope one of them is the JDBC driver
        final List<ProductArtifact> libArtifacts = jiraExecution.getProduct().getLibArtifacts();
        if (libArtifacts == null || libArtifacts.isEmpty()) {
            throw new MojoExecutionException(
                    "Product library artifact is empty, please provide library for database: " + databaseType);
        }
        return libArtifacts.stream()
                .map(this::toLibArtifact)
                .collect(toList());
    }

    private LibArtifact toLibArtifact(final ProductArtifact productArtifact) {
        return new LibArtifact(
                productArtifact.getGroupId(),
                productArtifact.getArtifactId(),
                productArtifact.getVersion()
        );
    }

    private void populateParameters(final DataSource dataSource) {
        if (isNotEmpty(defaultDatabase)) {
            dataSource.setDefaultDatabase(defaultDatabase);
        }
        if (isNotEmpty(systemUsername)) {
            dataSource.setSystemUsername(systemUsername);
        }
        if (isNotEmpty(systemPassword)) {
            dataSource.setSystemPassword(systemPassword);
        }
        if (isNotEmpty(dumpFilePath)) {
            dataSource.setDumpFilePath(dumpFilePath);
        }
        if (isNotEmpty(importMethod)) {
            dataSource.setImportMethod(importMethod);
        } else {
            // default is import standard sql
            dataSource.setImportMethod(SQL.getMethod());
        }
        getLog().info("Pre-integration-test import method: " + dataSource.getImportMethod());
    }

    private void dropAndCreate(final JiraDatabase jiraDatabase) throws MojoExecutionException {
        mojoExecutorWrapper.executeWithMergedConfig(
                getSqlMavenPlugin(jiraDatabase),
                goal("execute"),
                jiraDatabase.getSqlMavenCreateConfiguration(),
                getMavenContext().getExecutionEnvironment()
        );
    }

    private Plugin getSqlMavenPlugin(final JiraDatabase jiraDatabase) {
        final Plugin sqlMavenPlugin =
                getMavenContext().getPlugin("org.codehaus.mojo", "sql-maven-plugin");
        final List<Dependency> pluginDependencies = new ArrayList<>(sqlMavenPlugin.getDependencies());
        pluginDependencies.addAll(jiraDatabase.getSqlMavenDependencies());
        sqlMavenPlugin.setDependencies(pluginDependencies);
        return sqlMavenPlugin;
    }

    private void importDumpFile(final JiraDatabase jiraDatabase)
            throws MojoExecutionException {
        final DataSource dataSource = jiraDatabase.getDataSource();
        final String dumpFileName = dataSource.getDumpFilePath();
        if (isNotEmpty(dumpFileName)) {
            final File dumpFile = new File(dumpFileName);
            if (!dumpFile.isFile()) {
                throw new MojoExecutionException(format("Import file '%s' is not a file", dumpFileName));
            }
            getLog().info("Importing dump file: " + dumpFileName);
            if (SQL == ImportMethod.getValueOf(dataSource.getImportMethod())) {
                importSqlDumpFile(jiraDatabase);
            } else {
                importDataUsingDatabaseSpecificTool(jiraDatabase);
            }
        }
    }

    private void importSqlDumpFile(final JiraDatabase jiraDatabase) throws MojoExecutionException {
        mojoExecutorWrapper.executeWithMergedConfig(
                getSqlMavenPlugin(jiraDatabase),
                goal("execute"),
                jiraDatabase.getSqlMavenFileImportConfiguration(),
                getMavenContext().getExecutionEnvironment()
        );
    }

    private void importDataUsingDatabaseSpecificTool(JiraDatabase jiraDatabase) throws MojoExecutionException {
        final Xpp3Dom importConfiguration = jiraDatabase.getExecMavenToolImportConfiguration();
        if (importConfiguration == null) {
            getLog().warn(format("No configuration provided for %s - skipping import", jiraDatabase));
        } else {
            mojoExecutorWrapper.executeWithMergedConfig(
                    getMavenContext().getPlugin("org.codehaus.mojo", "exec-maven-plugin"),
                    goal("exec"),
                    importConfiguration,
                    getMavenContext().getExecutionEnvironment()
            );
        }
    }
}
