package com.atlassian.lesscss;

import com.google.common.collect.Lists;
import org.mozilla.javascript.*;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URI;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

public class RhinoLessCompiler implements LessCompiler {

    private final Scriptable sharedScope;

    public RhinoLessCompiler() {
        NotifyingContextFactory cf = new NotifyingContextFactory();
        ScriptableObjectSealer sealer = new ScriptableObjectSealer();
        cf.addListener(sealer);
        Context cx = cf.enterContext();
        try {
            ScriptableObject scope = cx.initStandardObjects();

            cx.setOptimizationLevel(9);
            try {
                loadJs(scope, cx, "/js/less/less-rhino.js");
                loadJs(scope, cx, "/js/less/less-patches.js");
            } catch (IOException e) {
                throw new IllegalStateException(e);
            }

            sealer.sealAll();
            cf.removeListener(sealer);

            sharedScope = scope;

        } finally {
            Context.exit();
        }

    }

    private void loadJs(Scriptable topScope, Context cx, String name) throws IOException {
        final InputStream in = getClass().getResourceAsStream(name);
        if (in == null) {
            throw new FileNotFoundException("Could not find JS resource " + name);
        }

        InputStreamReader reader = new InputStreamReader(in, "UTF-8");
        cx.evaluateReader(topScope, reader, name, 1, null);
    }

    @Override
    public String compile(Loader loader, URI uri, CharSequence content, boolean compress) {
        final ContextFactory cf = new ContextFactory();
        final Context cx = cf.enterContext();
        try {
            final Function runLessRun = (Function) sharedScope.get("runLessRun", sharedScope);

            Scriptable newScope = cx.newObject(sharedScope);
            newScope.setPrototype(sharedScope);
            newScope.setParentScope(null);

            try {
                final Object[] args = {uri, loader, content, compress};
                final Object result = runLessRun.call(cx, newScope, newScope, args);
                return Context.toString(result);
            } catch (JavaScriptException e) {
                throw newLessException(e);
            }
        } finally {
            Context.exit();
        }

    }

    private LessCompilationException newLessException(JavaScriptException e) {
        if (e.getValue() instanceof Scriptable) {
            Scriptable value = (Scriptable) e.getValue();
            String type = String.valueOf(ScriptableObject.getProperty(value, "type"));
            String message = String.valueOf(ScriptableObject.getProperty(value, "message"));
            if ("Syntax".equals(type)) {
                return new LessSyntaxException(message, e);
            } else if ("Unresolvable Import".equals(type)) {
                return new UnresolvableImportException(message, e);
            }
        }
        return new LessCompilationException(describeJsObject(e.getValue()), e);
    }

    private static String describeJsObject(Object jsObject) {
        try {
            Map<?, ?> map = (Map) Context.jsToJava(jsObject, Map.class);
            // LinkedHashMap has a json like toString output, which is good enough for us
            return new LinkedHashMap<>(map).toString();

        } catch (EvaluatorException e) {
            return Context.toString(jsObject);
        }
    }

    private static class ScriptableObjectSealer implements ScriptableCreationListener {

        private final List<ScriptableObject> objects = Lists.newArrayList();

        @Override
        public void onNewScriptable(Scriptable scriptable) {
            if (scriptable instanceof ScriptableObject) {
                objects.add((ScriptableObject) scriptable);
            }
        }

        public void sealAll() {
            for (ScriptableObject object : objects) {
                if (!object.isSealed()) {
                    object.sealObject();
                }
            }
        }
    }

}
