package com.atlassian.lesscss;

import org.mozilla.javascript.Context;
import org.mozilla.javascript.ContextFactory;
import org.mozilla.javascript.Scriptable;
import org.mozilla.javascript.ScriptableObject;

import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;

/**
 * A ContextFactory which notifies registered listeners of the creation of Scriptable objects
 */
class NotifyingContextFactory extends ContextFactory {

    private final Set<ScriptableCreationListener> creationListeners = new CopyOnWriteArraySet<>();

    @Override
    protected Context makeContext() {
        return new ListenableContext();
    }

    void addListener(ScriptableCreationListener listener) {
        creationListeners.add(listener);
    }

    void removeListener(ScriptableCreationListener listener) {
        creationListeners.remove(listener);
    }

    private void notifyCreationListeners(Scriptable value) {
        for (ScriptableCreationListener listener : creationListeners) {
            listener.onNewScriptable(value);
        }
    }

    class ListenableContext extends Context {
        ListenableContext() {
            super(NotifyingContextFactory.this);
        }

        @Override
        public Scriptable newArray(Scriptable scope, Object[] elements) {
            return notifyCreationListeners(super.newArray(scope, elements));
        }

        @Override
        public Scriptable newObject(Scriptable scope) {
            return notifyCreationListeners(super.newObject(scope));
        }

        @Override
        public Scriptable newObject(Scriptable scope, String constructorName) {
            return notifyCreationListeners(super.newObject(scope, constructorName));
        }

        @Override
        public Scriptable newObject(Scriptable scope, String constructorName, Object[] args) {
            return notifyCreationListeners(super.newObject(scope, constructorName, args));
        }

        @Override
        public Scriptable newArray(Scriptable scope, int length) {
            return notifyCreationListeners(super.newArray(scope, length));
        }

        @Override
        public ScriptableObject initStandardObjects(ScriptableObject scope, boolean sealed) {
            return notifyCreationListeners(super.initStandardObjects(scope, sealed));
        }

        private <T extends Scriptable> T notifyCreationListeners(T value) {
            NotifyingContextFactory.this.notifyCreationListeners(value);

            return value;
        }
    }

}
