package com.atlassian.lesscss;

import com.google.common.collect.ImmutableMap;
import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.imageio.ImageIO;
import java.awt.image.RenderedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Locale;
import java.util.Map;

public class DataUriUtils {

    private static final int MAX_BYTES = 32 * 1024; // IE8 has a 32KB data-uri limit

    private static final Logger log = LoggerFactory.getLogger(DataUriUtils.class);

    // A basic set of mime types to use. The consumer can specify a mime type explicitly if they wish so this
    // is good enough for now
    private static final Map<String, String> FILE_EXTENSION_TO_MIME_TYPE = ImmutableMap.<String, String>builder()
            .put("gif", "image/gif")
            .put("jpeg", "image/jpeg")
            .put("jpg", "image/jpeg")
            .put("png", "image/png")
            .put("svg", "image/svg+xml")
            .build();

    private DataUriUtils() {
        throw new UnsupportedOperationException();
    }

    public static String dataUri(String mimeType, byte[] bytes) {
        if (canOptimize(mimeType)) {
            bytes = optimize(bytes);
        }

        if (bytes.length >= MAX_BYTES) {
            log.info("Image exceeded limit and cannot be data-uri encoded. limit = {}, size = {}", MAX_BYTES, bytes.length);
            return null;
        }

        return "\"data:" + mimeType + ";base64," + base64Encode(bytes) + "\"";
    }

    public static String guessMimeType(String path) {
        String mimeType = "application/octet-stream";

        int dotIndex = path.lastIndexOf('.');
        if (dotIndex != -1) {
            String extension = path.substring(dotIndex + 1).toLowerCase(Locale.US);
            String matchingMimeType = FILE_EXTENSION_TO_MIME_TYPE.get(extension);
            if (matchingMimeType != null) {
                mimeType = matchingMimeType;
            }
        }

        return mimeType;
    }

    private static String base64Encode(byte[] bytes) {
        return new String(Base64.encodeBase64(bytes));
    }

    private static boolean canOptimize(String mimeType) {
        return "image/png".equals(mimeType);
    }

    private static byte[] optimize(byte[] bytes) {
        try {
            RenderedImage image = ImageIO.read(new ByteArrayInputStream(bytes));
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            ImageIO.write(image, "png", out);
            return out.toByteArray();
        } catch (IOException e) {
            log.info("Failed to optimise image. Falling back to unoptimised version", e);
            return bytes;
        }
    }

}
