// less-rhino logs to System.out and System.err. We don't want it to do this
console = (function() {
    // For now, we don't want to log anything.
    // We may want to hook this up to an SLF4j logger in the future
    function noop() {
    }

    return {
        log: noop,
        info: noop,
        error: noop,
        warn: noop
    };
})();

function ensureString(s) {
    return s + ''; // ensure java String turns into js string
}

// override the fs module. We don't want less trying to read from the filesystem.
less.modules.fs = {};

// Override parseEnv and evalEnv so that our env cascades down correctly
(function(tree) {

    function EmptyConstructor() {

    }

    function createPrototype(SuperConstructor) {
        EmptyConstructor.prototype = SuperConstructor.prototype;
        try {
            return new EmptyConstructor();
        } finally {
            EmptyConstructor.prototype = null;
        }
    }

    tree.parseEnv = (function(originalParseEnv) {
        function parseEnv(options) {
            originalParseEnv.call(this, options);
            this.alreadySeen = options.alreadySeen;
            this.ourLoader = options.ourLoader;
        }

        //noinspection JSPotentiallyInvalidConstructorUsage
        parseEnv.prototype = createPrototype(originalParseEnv);

        return parseEnv;
    })(tree.parseEnv);

    tree.evalEnv = (function(originalEvalEnv) {
        function evalEnv(options, frames) {
            originalEvalEnv.call(this, options, frames);
            this.ourLoader = options && options.ourLoader;
        }

        //noinspection JSPotentiallyInvalidConstructorUsage
        evalEnv.prototype = createPrototype(originalEvalEnv);

        return evalEnv;
    })(tree.evalEnv);

})(less.tree);

// less.tree.defaultFunc stores state against itself which causes issues in a multi-threaded environment
less.tree.defaultFunc = (function(originalDefaultFunc) {
    function DefaultFunc() {

    }
    DefaultFunc.prototype = originalDefaultFunc;


    var threadLocal = new java.lang.ThreadLocal();
    var threadSafeDefaultFunc = {};
    for (var funcName in originalDefaultFunc) {
        if (Object.prototype.hasOwnProperty.call(originalDefaultFunc, funcName)) {
            (function(funcName) {
                threadSafeDefaultFunc[funcName] = function(arg) {
                    var defaultFunc = threadLocal.get();
                    return defaultFunc[funcName].call(defaultFunc, arg);
                };
            })(funcName);
        }
    }
    threadSafeDefaultFunc.setupThreadState = function() {
        threadLocal.set(new DefaultFunc());
    };
    threadSafeDefaultFunc.destroyThreadState = function() {
        threadLocal.remove();
    };

    return threadSafeDefaultFunc;
})(less.tree.defaultFunc);

(function(less){

    // custom functions (all names are lowercased before lookup in tree.functions)
    less.tree.functions.encodeuricomponent = function(a) {
        var str = encodeURIComponent(a.value ? a.value : a.toCSS());
        return new (tree.Quoted)('"' + str + '"', str);
    };
    less.tree.functions.encodeuri = function(a) {
        var str = encodeURI(a.value ? a.value : a.toCSS());
        return new (tree.Quoted)('"' + str + '"', str);
    };

    // Patch 'data-uri' function
    var _loadDataUri = function(mimeTypeNode, filePathNode, makeAbsolute) {
        if (!filePathNode) {
            filePathNode = mimeTypeNode;
            mimeTypeNode = null;
        }

        var path = filePathNode && filePathNode.value;
        var mimeType = mimeTypeNode && mimeTypeNode.value;

        var baseUri = new java.net.URI(this.currentFileInfo.filename);
        if (makeAbsolute && !/^\//.test(path)) {
            path = "/" + path;
        }
        var uri = this.env.ourLoader.resolve(baseUri, path);

        if (uri) {
            var dataUri = this.env.ourLoader.dataUri(mimeType, uri);
            if (dataUri) {
                return new(less.tree.URL)(new(less.tree.Anonymous)(ensureString(dataUri)));
            }
        }

        return new(less.tree.URL)(filePathNode, this.currentFileInfo).eval(this.env);
    };
    less.tree.functions['data-uri'] = function(mimeTypeNode, filePathNode) {
        return _loadDataUri.call(this, mimeTypeNode, filePathNode, false);
    };

    // Add backwards compatibility for JIRA's function names
    less.tree.functions.inlineimage = less.tree.functions.inlineimageunoptim = function(mimeTypeNode, filePathNode) {
        return _loadDataUri.call(this, mimeTypeNode, filePathNode, true);
    };
})(less);

// less.tree.visitor lazily initialises some global state. Create a visitor to initialise it
new less.tree.visitor(null);

var runLessRun = (function(less) {

    // We need to use a custom importer so that less loads files from the correct location
    less.Parser.importer = function(path, currentFileInfo, callback, env) {
        var baseUri = new java.net.URI(currentFileInfo.filename);
        // First resolve the uri based on the current baseUri
        var uri;
        try {
            uri = env.ourLoader.resolve(baseUri, path);
        } catch (e) {
            callback(e);
            return;
        }

        if (!uri) {
            callback({
                type: 'Unresolvable Import',
                message: 'Unable to resolve import ' + path
            });
            return;
        }

        var data;
        if (env.alreadySeen[uri]) {
            data = "/* skipping already included " + uri + " */\n";
        } else {
            env.alreadySeen[uri] = true;

            data = env.ourLoader.load(uri);
            if (data == null) {
                callback({
                    type: 'Unresolvable Import',
                    message: 'Cannot find ' + path
                });
                return;
            }
            data = ensureString(data);
        }

        var newEnv = new less.tree.parseEnv(env);

        newEnv.contents[uri] = data;
        newEnv.currentFileInfo = createFileInfo(uri);
        newEnv.processImports = false;


        var parser = new less.Parser(newEnv);
        parser.parse(data, function(e, root) {
            if (e) {
                callback(e);
            } else {
                callback(e, root, ensureString(uri));
            }
        });
    };

    function createFileInfo(uri) {
        var absolutePath = ensureString(uri);
        var relativePath = ensureString(uri.getPath());
        var absoluteDirectory = absolutePath.substring(0, absolutePath.lastIndexOf('/'));

        return  {
            relativeUrls: true,
            filename: absolutePath,
            rootpath: null, // This is used to resolve relative urls in url(). We want it to leave it as is
            currentDirectory: absoluteDirectory
        };
    }

    function withThreadLocal(func) {
        return function() {
            less.tree.defaultFunc.setupThreadState();
            try {
                return func.apply(this, arguments);
            } finally {
                less.tree.defaultFunc.destroyThreadState();
            }
        };
    }

    // A wrapper around the less parser
    function runLessRun(baseUri, loader, lessToCompile, compress) {
        var alreadySeen = {};
        var env = {
            alreadySeen: alreadySeen,
            currentFileInfo: createFileInfo(baseUri),
            filename: ensureString(baseUri),
            javascriptEnabled: false,
            ourLoader: loader,
            relativeUrls: true
        };
        var parser = new less.Parser(env);

        var result = '';
        var error = null;
        parser.parse(ensureString(lessToCompile), function (e, root) {
            if (e) {
                error = e;
            } else {
                result = root.toCSS({
                    compress: compress,
                    ourLoader: loader
                });
            }
        });

        if (error) {
            throw error;
        }

        return result;
    }

    return withThreadLocal(runLessRun);
})(less);