AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:test-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer");

module("JIRA.WorkflowDesigner.TransitionConnection", {
    /**
     * Check that all the ports in a status are visible
     *
     * @param {JIRA.WorkflowDesigner.StatusView} statusView Status view that owns the ports
     * @returns {boolean} Whether all ports are visible
     */
    allPortsAreVisible: function (statusView) {
        return this.checkPortsVisibility(statusView, []);
    },

    /**
     * Check if all the ports not defined in hiddenPorts are visible. Also checks
     * that all ports defined in hiddenPorts are not visible.
     *
     * Example:
     *
     * checkPortsVisibility(status, [0,1,2]) returns true if ports 3, 4, 5, 6 and 7 are visible and
     * ports 0, 1 and 2 are invisible
     *
     * @param {JIRA.WorkflowDesigner.StatusView} statusView Status view that owns the ports
     * @param {number[]} [hiddenPorts] Indexes of ports to check for invisibility
     * @returns {boolean} Whether all ports are visible
     */
    checkPortsVisibility: function (statusView, hiddenPorts) {
        return _.all(statusView.getPorts(), function (port, index) {
            if (_.contains(hiddenPorts, index)) {
                return !port.isVisible();
            }
            return port.isVisible();
        });
    },

    createTestData: function(initial) {
        this.sourceView = new JIRA.WorkflowDesigner.StatusView({
            canvas: this.canvas,
            model: new JIRA.WorkflowDesigner.StatusModel({x: 10, y: 0}),
            workflowModel: new JIRA.WorkflowDesigner.WorkflowModel(),
            isPortDragged: jQuery.noop,
            getAllTargetPorts: _.bind(this._getAllPorts, this)
        }).render();

        this.targetView = new JIRA.WorkflowDesigner.StatusView({
            canvas: this.canvas,
            model: new JIRA.WorkflowDesigner.StatusModel({x: 200, y: 0 }),
            workflowModel: new JIRA.WorkflowDesigner.WorkflowModel(),
            isPortDragged: jQuery.noop,
            getAllTargetPorts: _.bind(this._getAllPorts, this)
        }).render();

        this.transitionConnection = new JIRA.WorkflowDesigner.TransitionConnection(!!initial);
        this.transitionConnection.installEditPolicy(new JIRA.WorkflowDesigner.Policy.Feedback.LineSelectionFeedbackPolicy());
        this.transitionConnection.setSource(this.sourceView.getPorts()[0]);
        this.transitionConnection.setTarget(this.targetView.getPorts()[0]);

        this.canvas.addFigure(this.transitionConnection);
    },

    _getAllPorts: function() {
        return _.union(this.sourceView.getPorts(), this.targetView.getPorts());
    },

    selectTransition: function () {
        this.canvas.selectFigure(this.transitionConnection);
    },

    setup: function () {
        function getCanvasBoundingBox() {
            return new draw2d.geo.Rectangle(0, 0, 0, 0);
        }

        this.sandbox = sinon.sandbox.create();
        this.canvas = JIRA.WorkflowDesigner.TestUtilities.testDraw2DCanvas();
        this.selectionPolicy = new JIRA.WorkflowDesigner.Policy.Canvas.PanningSingleSelectionPolicy(getCanvasBoundingBox);
    },

    teardown: function () {
        this.sandbox.restore();
    }
});

test("Changing source port should relocate the transition label", function () {
    var relocateStub;

    this.createTestData();

    relocateStub = this.sandbox.stub(JIRA.WorkflowDesigner.Layout.Locator.ManhattanPercentageLocator.prototype, "relocate");

    this.transitionConnection.setSource(this.sourceView.getPorts()[0]);

    equal(relocateStub.callCount, 1, "relocate() should be called once");
});

test("Changing target port should relocate the transition label", function () {
    var relocateStub;

    this.createTestData();

    relocateStub = this.sandbox.stub(JIRA.WorkflowDesigner.Layout.Locator.ManhattanPercentageLocator.prototype, "relocate");

    this.transitionConnection.setTarget(this.sourceView.getPorts()[0]);

    equal(relocateStub.callCount, 1, "relocate() should be called once");
});

test("Label text is truncated if maximum length is exceeded", function() {
    var connection = new JIRA.WorkflowDesigner.TransitionConnection();

    connection._getEllipsis = sinon.stub().returns("\u2026");

    connection.setText(null);
    equal(connection._label.getText(), undefined, "Null label stays null");

    connection.setText("12345");
    equal(connection._label.getText(), "12345", "Label shorter than maximum is not truncated");

    connection.setText("123456789 123456789 123456789 ");
    equal(connection._label.getText(), "123456789 123456789 123456789 ", "Label equal to maximum is not truncated");

    connection.setText("123456789 123456789 123456789 123456789");
    equal(connection._label.getText(), "123456789 123456789 123456789\u2026", "Label longer than maximum is truncated and ellipsis is appended");
});

test("Transitions should have a gap", function () {
    var transition = new JIRA.WorkflowDesigner.TransitionConnection(false);

    sinon.stub(transition, "getTarget").returns({
        getConnectionDirection: function () {
            return JIRA.WorkflowDesigner.Direction.RIGHT;
        }
    });
    sinon.stub(transition, "getSource").returns({
        getConnectionDirection: function () {
            return JIRA.WorkflowDesigner.Direction.RIGHT;
        }
    });

    deepEqual(transition.getEndPointGapTransformation(), {x: 3, y: 0}, "The end point should have a gap of 3 pixels");
    deepEqual(transition.getStartPointGapTransformation(), {x: 3, y: 0}, "The start point should have a gap of 3 pixels");
});

test("Global Transitions should have a gap only in the endpoint", function () {
    var transition = new JIRA.WorkflowDesigner.TransitionConnection(true);

    sinon.stub(transition, "getTarget").returns({
        getConnectionDirection: function () {
            return JIRA.WorkflowDesigner.Direction.RIGHT;
        }
    });
    sinon.stub(transition, "getSource").returns({
        getConnectionDirection: function () {
            return JIRA.WorkflowDesigner.Direction.RIGHT;
        }
    });

    deepEqual(transition.getEndPointGapTransformation(), {x: 3, y: 0}, "The end point should have a gap of 3 pixels");
    deepEqual(transition.getStartPointGapTransformation(), {x: 0, y: 0}, "The start point should have a gap of 3 pixels");
});

test("Status target ports are shown when connection start/end is dragged", function () {
    var endResizeHandle, selectionHandles, startResizeHandle;
    this.createTestData();

    this.selectTransition();
    selectionHandles = this.transitionConnection.selectionHandles;
    startResizeHandle = selectionHandles.get(0);
    endResizeHandle = selectionHandles.get(1);

    ok(jQuery(startResizeHandle.shape).is(":visible"), "Start resize handler is visible");
    ok(jQuery(endResizeHandle.shape).is(":visible"), "End resize handler is visible");

    ok(!this.allPortsAreVisible(this.sourceView), "All ports in the source are not visible before drag");
    ok(!this.allPortsAreVisible(this.targetView), "All ports in the target are not visible before drag");

    startResizeHandle.onDragStart();
    ok(this.allPortsAreVisible(this.sourceView), "When resizing from the start, all ports in the source are visible on drag");
    ok(!this.allPortsAreVisible(this.targetView), "When resizing from the start, all ports in the target are not visible on drag");

    startResizeHandle.onDragEnd();
    ok(!this.allPortsAreVisible(this.sourceView), "When resizing from the start, all ports in the source are not visible after drag");
    ok(!this.allPortsAreVisible(this.targetView), "When resizing from the start, all ports in the target are not visible after drag");

    endResizeHandle.onDragStart();
    ok(this.checkPortsVisibility(this.sourceView, [0]), "When resizing from the end, all ports in the source except the opposite port are visible on drag");
    ok(this.allPortsAreVisible(this.targetView), "When resizing from the end, all ports in the target are visible on drag");

    endResizeHandle.onDragEnd();
    ok(!this.allPortsAreVisible(this.sourceView), "When resizing from the end, all ports in the source are not visible after drag");
    ok(!this.allPortsAreVisible(this.targetView), "When resizing from the end, all ports in the target are not visible after drag");
});

test("Connection's port is hidden when the opposite port is dragged", function () {
    var status,
        transition,
        selectionHandles,
        startResizeHandle,
        endResizeHandle;

    status = new JIRA.WorkflowDesigner.StatusView({
        canvas: this.canvas,
        model: new JIRA.WorkflowDesigner.StatusModel({x: 10, y: 0}),
        workflowModel: new JIRA.WorkflowDesigner.WorkflowModel(),
        isPortDragged: jQuery.noop,
        getAllTargetPorts: function() {
            return status.getPorts();
        }
    }).render();

    transition = new JIRA.WorkflowDesigner.TransitionConnection(false);
    transition.installEditPolicy(new JIRA.WorkflowDesigner.Policy.Feedback.LineSelectionFeedbackPolicy());
    transition.setSource(status.getPorts()[0]);
    transition.setTarget(status.getPorts()[1]);
    this.canvas.addFigure(transition);
    this.canvas.selectFigure(transition);

    selectionHandles = transition.selectionHandles;
    startResizeHandle = selectionHandles.get(0);
    endResizeHandle = selectionHandles.get(1);

    startResizeHandle.onDragStart();
    ok(this.checkPortsVisibility(status, [1]), "When resizing from the start, all ports except the port #1 are visible");
    startResizeHandle.onDragEnd();

    endResizeHandle.onDragStart();
    ok(this.checkPortsVisibility(status, [0]), "When resizing from the start, all ports except the port #0 are visible");
    endResizeHandle.onDragEnd();
});

test("The label is always visible on the Initial transitions", function () {
    this.createTestData(true);

    ok(this.transitionConnection.textIsVisible(), "The transition's label is visible");
});

test("Start resize handler is not visible on Initial transition", function () {
    this.createTestData(true);
    this.selectTransition();

    ok(!jQuery(this.transitionConnection.selectionHandles.get(0).shape).is(":visible"), "Start resize handler is invisible");
    ok(jQuery(this.transitionConnection.selectionHandles.get(1).shape).is(":visible"), "End resize handler is visible");
});

test("Clicking a transition's label selects the transition", function () {
    this.createTestData();

    sinon.stub(this.canvas, "getBestFigure").returns(this.transitionConnection._label);
    this.selectionPolicy.onMouseDown(this.canvas, 0, 0);

    ok(this.canvas.getSelectedFigure() === this.transitionConnection, "The transition was selected");
});

test("Line start handle's onDragEnd operation does nothing when moused over a drop target and dropped on nothing", function () {
    var lineStartHandle, originalPort;

    this.createTestData();
    this.selectTransition();

    originalPort = this.sourceView.getPorts()[0];

    lineStartHandle = this.transitionConnection.selectionHandles.get(0);
    lineStartHandle.onDragStart();

    // getDropTarget returns sourceView's second port because the cursor is currently over it
    var stubbedGetDropTarget = this.sandbox.stub(lineStartHandle, "_getDropTarget").returns(this.sourceView.getPorts()[1]);
    // Drag the handle, setting the current drop target
    lineStartHandle.onDrag(0, 0);

    // getDropTarget returns null because cursor is not over a port
    stubbedGetDropTarget.returns(null);
    lineStartHandle.onDrag(0, 0);
    lineStartHandle.onDragEnd();

    ok(this.transitionConnection.getSource() === originalPort, "The source port should revert back to the original port.");
});

test("Line start handle's onDragEnd undoes operation when dropped on a drop target", function () {
    var lineStartHandle, mousedOverPort;

    this.createTestData();
    this.selectTransition();

    mousedOverPort = this.sourceView.getPorts()[1];

    lineStartHandle = this.transitionConnection.selectionHandles.get(0);
    lineStartHandle.onDragStart();

    // getDropTarget returns sourceView's second port because the cursor is currently over it
    this.sandbox.stub(lineStartHandle, "_getDropTarget").returns(mousedOverPort);
    lineStartHandle.onDrag(0, 0);
    lineStartHandle.onDragEnd();

    ok(this.transitionConnection.getSource() === mousedOverPort, "The source after drag end should be the moused over port.");
});

test("#positionArrow sets the target decorator's z-index", function () {
    var insertAfterSpy;

    this.createTestData();
    insertAfterSpy = sinon.spy(this.transitionConnection.targetDecoratorNode, "insertAfter");

    this.transitionConnection.positionArrow();
    equal(insertAfterSpy.callCount, 1, "The target decorator's insertAfter method was called");
    ok(insertAfterSpy.args[0][0] === this.transitionConnection.shape, "It was passed the correct argument");
});