AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:test-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer");

module("JIRA.WorkflowDesigner.Dialogs.AddStatusSuggestionHandler", {
    /**
     * Create an `AJS.SelectModel` containing the given statuses.
     *
     * @param {object[]} statuses The status objects to be contained in the `SelectModel`.
     * @returns {AJS.SelectModel} An `AJS.SelectModel` containing `statuses`.
     */
    createSelectModel: function (statuses) {
        var select;

        select = _.reduce(statuses, function (select, status) {
            return select.append(jQuery("<option>").text(status.name).val(status.id));
        }, jQuery("<select>"));

        return new AJS.SelectModel({
            element: select
        });
    },

    /**
     * Create an `AddStatusSuggestionHandler`.
     *
     * @param {object} [options] Options to pass to the constructor.
     * @param {AJS.SelectModel} [options.selectModel] The model to pass to the constructor.
     * @returns {JIRA.WorkflowDesigner.Dialogs.AddStatusSuggestionHandler} A selection handler.
     */
    createSuggestionHandler: function (options) {
        var statuses;

        statuses = [{
            id: "closed",
            name: "Closed"
        }, {
            id: "in-progress",
            name: "In Progress"
        }, {
            id: "open",
            name: "Open"
        }];

        options = _.defaults({}, options, {
            selectModel: this.createSelectModel(statuses.slice(0, -1)),
            statuses: statuses
        });

        return new JIRA.WorkflowDesigner.Dialogs.AddStatusSuggestionHandler(options, options.selectModel);
    },

    setup: function () {
        this.suggestionHandler = this.createSuggestionHandler();
    }
});

test("A \"create status\" option is shown in a separate group", function () {
    this.suggestionHandler.execute("New Status").done(function (descriptors) {
        equal(descriptors.length, 2, "Two groups were returned");
        equal(descriptors[1].items().length, 1, "The second group contains a single item");
        ok(!descriptors[1].items()[0].value(), "The item represents a new status");
    });
});

test("No \"create status\" option is shown if the query is empty", function () {
    this.suggestionHandler.execute("").done(function (descriptors) {
        equal(descriptors.length, 1, "A single group was returned");
        equal(descriptors[0].items().length, 1, "It contains a single item");
        equal(descriptors[0].items()[0].value(), "in-progress", "The item's value is correct");
    });
});

test("No \"create status\" option is shown if the query matches an existing status", function () {
    // The check should be case and whitespace insensitive.
    var queries = ["open", "Open", "In Progress", " In Progress "];

    _.each(queries, function (query) {
        this.suggestionHandler.execute(query).done(function (descriptors) {
            equal(descriptors.length, 1, "A single group was returned");
            equal(descriptors[0].items().length, 1, "It contains a single item");
            equal(descriptors[0].items()[0].value(), "in-progress", "The item's value is correct");
        });
    }, this);
});

test("Only the \"create status\" group is returned if there are no existing statuses", function () {
    this.suggestionHandler = this.createSuggestionHandler({
        selectModel: this.createSelectModel([])
    });

    this.suggestionHandler.execute("New Status").done(function (descriptors) {
        equal(descriptors.length, 1, "A single group was returned");
        equal(descriptors[0].items().length, 1, "It contains a single item");
        ok(!descriptors[0].items()[0].value(), "The item represents a new status");
    });
});