(function (Backbone) {
    /**
     * @class JIRA.WorkflowDesigner.Dialogs.EditTransitionDialogView
     * @classdesc The workflow designer's "Edit Transition" dialog.
     * @extends JIRA.WorkflowDesigner.Dialogs.TransitionDialogView
     */
    Backbone.define("JIRA.WorkflowDesigner.Dialogs.EditTransitionDialogView", JIRA.WorkflowDesigner.Dialogs.TransitionDialogView.extend(
    /** @lends JIRA.WorkflowDesigner.Dialogs.EditTransitionDialogView# */
    {
        id: "edit-transition-dialog",

        template: JIRA.WorkflowDesigner.Templates.editTransitionDialog,

        ui: {
            description: "#transition-description",
            name: "#transition-name",
            screenId: "#transition-screen-id"
        },

        /**
         * @returns {function} A function that retrieves the dialog's content.
         * @private
         */
        _getContent: function () {
            var instance = this;

            return function (callback) {
                var data = instance._transitionModel.toJSON(),
                    request = instance._getScreenOptions();

                request.done(function (screenOptions) {
                    _.extend(data, {screens: screenOptions});
                    callback(instance.template(data));
                });

                request.fail(instance._onLoadError);
            };
        },

        /**
         * @returns {string} The ID of the selected screen.
         * @private
         */
        _getSelectedScreenId: function () {
            return Number(this.ui.screenId.find(":selected").val());
        },

        /**
         * @returns {string} The name of the selected screen.
         * @private
         */
        _getSelectedScreenName: function () {
            return this.ui.screenId.find(":selected").text();
        },

        /**
         * Make a request to update the transition.
         *
         * @return {jQuery.Deferred} A deferred that is resolved on success.
         * @private
         */
        _makeRequest: function (options) {
            return JIRA.WorkflowDesigner.IO.AJAX.WorkflowTransitionsAJAXManager.updateTransition(options);
        },

        /**
         * @private
         */
        _onContentRefresh: function () {
            new AJS.SingleSelect({
                element: this.ui.screenId,
                revertOnInvalid: true
            });
        },

        /**
         * @param {jQuery.Event} e The submit event.
         * @param {function} callback A function to call when submission is complete.
         * @private
         */
        _onSubmit: function (e, callback) {
            var options, request;

            options = {
                description: this.ui.description.val(),
                name: this.ui.name.val(),
                screenId: this._getSelectedScreenId(),
                sourceStepId: this._transitionModel.getSourceStepId(),
                transitionId: this._transitionModel.getActionId(),
                workflowName: this._workflowModel.get("name")
            };

            JIRA.WorkflowDesigner.Dialogs.TransitionDialogView.prototype._onSubmit.apply(this, arguments);
            request = this._makeRequest(options);
            request.done(this._onSuccess).fail(callback).fail(this._onError);
        },

        /**
         * @private
         */
        _onSuccess: function() {
            this.hide();
            this._transitionModel.set({
                description: this.ui.description.val(),
                name: this.ui.name.val(),
                screenId: this._getSelectedScreenId(),
                screenName: this._getSelectedScreenName()
            });

            this._triggerAnalyticsEvent();
        },

        /**
         * Trigger an analytics event indicating that the update was successful.
         *
         * @protected
         */
        _triggerAnalyticsEvent: function () {
            JIRA.WorkflowDesigner.Analytics.triggerUpdateTransition();
        }
    }));
}(JIRA.WorkflowDesigner.Backbone));