AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:test-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer");

module("JIRA.WorkflowDesigner.Draw2DCanvas", {
    setup: function () {
        this.canvas = JIRA.WorkflowDesigner.TestUtilities.testDraw2DCanvas();
        this.singleSelectionPolicy = new draw2d.policy.canvas.SingleSelectionPolicy();
        this.canvas.installEditPolicy(this.singleSelectionPolicy);
        this.sandbox = sinon.sandbox.create();
        this.sandbox.stub(JIRA.WorkflowDesigner.BrowserUtilities, "getZoomLevel");
        this.sandbox.stub(JIRA.Messages, "showErrorMsg").returns({
            css: jQuery.noop,
            remove: jQuery.noop
        });
    },

    teardown: function () {
        this.sandbox.restore();
    }
});

test("createLayers()", function () {
    var Layer = JIRA.WorkflowDesigner.Draw2DCanvas.Layer,
        layer;

    // Creating a single layer.
    equal(this.canvas.getLayer("a"), undefined, "Layer a is undefined");

    this.canvas.createLayers("a");
    layer = this.canvas.getLayer("a");
    ok(layer instanceof Layer, "Layer a is defined");

    this.canvas.createLayers("a");
    ok(this.canvas.getLayer("a") === layer, "Calling createLayers() is idempotent");

    // Creating multiple layers.
    equal(this.canvas.getLayer("b"), undefined, "Layer b is undefined");
    equal(this.canvas.getLayer("c"), undefined, "Layer c is undefined");

    this.canvas.createLayers("b", "c");
    ok(this.canvas.getLayer("b") instanceof Layer, "Layer b is defined");
    ok(this.canvas.getLayer("c") instanceof Layer, "Layer c is defined");
});

test("Double clicks outside the canvas aren't observed", function () {
    var onDoubleClickStub = this.sandbox.stub(this.canvas, "onDoubleClick");

    // Don't attempt to calculate the location of the double click.
    this.canvas.fromDocumentToCanvasCoordinate = function () {
        return new draw2d.geo.Point(0, 0);
    };

    jQuery(document).dblclick();
    equal(onDoubleClickStub.callCount, 0, "The double click wasn't observed");

    this.canvas.html.dblclick();
    equal(onDoubleClickStub.callCount, 1, "The double click was observed");
});

test("Dragging outside the container", function () {
    var onMouseDownSpy = sinon.spy(this.singleSelectionPolicy, "onMouseDown"),
        onMouseDragSpy = sinon.spy(this.singleSelectionPolicy, "onMouseDrag");

    // Trigger a mouse down event on the canvas.
    this.canvas.html.trigger(jQuery.Event("mousedown", {clientX: 0, clientY: 0, which: 0}));
    equal(onMouseDownSpy.callCount, 1, "The mouse down event was handled");

    // Trigger a mouse move event on the document, not the canvas.
    jQuery(document).trigger(jQuery.Event("mousemove", {clientX: 0, clientY: 0}));
    equal(onMouseDragSpy.callCount, 1, "The mouse move event was handled");
});

test("fitToContainer()", function () {
    var setSizeStub = sinon.stub(this.canvas.paper, "setSize");

    this.canvas.html.css({height: 100, width: 100});
    this.canvas.svgElement.attr({height: 100, width: 100});
    this.canvas.fitToContainer();
    equal(setSizeStub.callCount, 0, "Does nothing if the elements are the same size");

    this.canvas.html.css("height", 200).hide();
    this.canvas.fitToContainer();
    equal(setSizeStub.callCount, 0, "Does nothing if the SVG element isn't visible");

    this.canvas.html.show();
    this.canvas.fitToContainer();
    equal(setSizeStub.callCount, 1, "Paper#setSize() was called to resize the SVG element");
    deepEqual(setSizeStub.args[0], [100, 200], "It was passed the correct arguments");
    equal(this.canvas.svgElement.css("height"), "200px", "The SVG element's CSS height is correct");
    equal(this.canvas.svgElement.css("width"), "100px", "The SVG element's CSS width is correct");
});

test("getLayer()", function () {
    var layer;

    layer = this.canvas.getLayer("layer");
    equal(layer, undefined, "getLayer() returns undefined for unknown layers");

    this.canvas.createLayers("layer");
    layer = this.canvas.getLayer("layer");
    ok(layer instanceof JIRA.WorkflowDesigner.Draw2DCanvas.Layer,
            "getLayer() returns defined layers");
});

test("getSelectedFigure()", function () {
    var figure = new draw2d.shape.basic.Rectangle();
    this.canvas.addFigure(figure);

    equal(this.canvas.getSelectedFigure(), null, "The selected figure is initially null");

    this.singleSelectionPolicy.select(this.canvas, figure);
    ok(this.canvas.getSelectedFigure() === figure, "getSelectedFigure() returns the selected figure");
});

test("removeFigure() removes the figure from its layer", function () {
    var figure = new draw2d.shape.basic.Rectangle(),
        layer;

    this.canvas.createLayers("layer");
    layer = this.canvas.getLayer("layer");
    layer.addFigure(figure);
    sinon.spy(layer, "removeFigure");

    this.canvas.removeFigure(figure);
    equal(layer.removeFigure.callCount, 1, "Layer#removeFigure() was called");
    ok(layer.removeFigure.args[0][0] === figure, "It was passed the correct figure");
});

test("Removing the currently hovered figure clears the currently hovered figure", function () {
    var figure = new draw2d.shape.basic.Rectangle();

    this.canvas.addFigure(figure);
    this.canvas.currentHoverFigure = figure;

    this.canvas.removeFigure(figure);
    equal(this.canvas.currentHoverFigure, null, "The currently hovered figure was cleared");
});

test("selectFigure()", function () {
    var figure = new draw2d.shape.basic.Rectangle();
    this.canvas.addFigure(figure);

    this.canvas.selectFigure(figure);
    ok(this.canvas.getSelection().getPrimary() === figure, "The figure was selected");

    this.canvas.selectFigure(null);
    equal(this.canvas.getSelection().getPrimary(), null, "The selection was cleared");
});
