(function () {
    /**
     * Current step id
     *
     * @inner
     * @type {number}
     */
    var currentStepId;

    /**
     * Cache for layout data
     *
     * @inner
     * @type {object}
     */
    var layoutData;

    /**
     * Dialog
     *
     * @inner
     * @type {JIRA.WorkflowDesigner.Dialog}
     */
    var workflowDialog;

    /**
     * Name of the workflow
     *
     * @inner
     * @type {string}
     */
    var workflowName;

    /**
     * Extract a parameter from a workflow image URL.
     *
     * @inner
     * @param {string} imageURL The workflow image URL.
     * @param {string} key The key of the parameter to extract.
     * @return {string} The value corresponding to <tt>key</tt> in <tt>imageURL</tt>'s query string.
     */
    function getImageURLParameter(imageURL, key) {
        var value = parseUri(imageURL).queryKey[key];
        return decodeURIComponent(value.replace(/\+/g, "%20"));
    }

    /**
     * Load the layout data of the current issue's workflow.
     *
     * @inner
     * @return {jQuery.Deferred} A deferred that is resolved when the layout data has loaded.
     */
    function loadLayoutData() {
        var request;

        if (layoutData) {
            return jQuery.Deferred().resolve(layoutData).promise();
        } else {
            request = JIRA.WorkflowDesigner.IO.AJAX.WorkflowAJAXManager.load(workflowName);
            request.done(function (models, response) {
                layoutData = response;
            });

            return request;
        }
    }

    AJS.namespace("JIRA.WorkflowDesigner.ViewIssue");

    /**
     * Code related to showing a read-only version of the new workflow designer on the view issue page.
     *
     * @namespace
     */
    JIRA.WorkflowDesigner.ViewIssue = {
        /**
         * Patch the "View Workflow" link to open the new workflow designer.
         *
         * Does nothing if the link has already been patched.
         *
         * @param {element} context The context in which to look for the link.
         */
        replaceViewWorkflowLink: function (context) {
            var template = JIRA.WorkflowDesigner.Templates.ViewIssue.viewWorkflowLink,
                viewWorkflowLink,
                workflowImageURL;

            context = jQuery(context);
            viewWorkflowLink = context.find(".issueaction-viewworkflow:not(.new-workflow-designer)");
            workflowImageURL = viewWorkflowLink.attr("href");

            if (viewWorkflowLink.length) {
                currentStepId = parseInt(getImageURLParameter(workflowImageURL, "stepId"), 10);
                layoutData = null;
                viewWorkflowLink.replaceWith(template({href: workflowImageURL}));
                workflowName = getImageURLParameter(workflowImageURL, "workflowName");
            }
        }
    };

    // Replace the old "View Workflow" link on NEW_CONTENT_ADDED.
    JIRA.bind(JIRA.Events.NEW_CONTENT_ADDED, function (e, context) {
        if (JIRA.WorkflowDesigner.browserIsSupported()) {
            JIRA.WorkflowDesigner.ViewIssue.replaceViewWorkflowLink(context);
        }
    });

    workflowDialog = new JIRA.WorkflowDesigner.Dialog({
        content: function (callback) {
            var layoutDataRequest = loadLayoutData();

            var dialog = this;
            layoutDataRequest.done(function () {
                dialog.createDialogContent(callback);
            });

            layoutDataRequest.fail(function (errorMessage) {
                JIRA.WorkflowDesigner.Messages.showErrorMessage(errorMessage);
                workflowDialog.hide(true);
            });
        },
        id: "view-workflow-dialog",
        trigger: ".issueaction-viewworkflow.new-workflow-designer",
        workflowId: function() {
            return workflowName;
        },
        currentStepId: function() {
            return currentStepId;
        },
        layoutData: function() {
            return JIRA.WorkflowDesigner.IO.WorkflowDataReader.read(layoutData);
        }
    });
}());