(function () {

    /**
     * Duration of the animation
     *
     * @inner
     * @constant
     * @type {number}
     * @default
     */
    var animationDuration = 500;

    /**
     * Original properties key
     *
     * @inner
     * @constant
     * @type {string}
     * @default
     */
    var originalPropertiesKey = "full-screen-controller-original-properties";

    /**
     * Placeholder key
     *
     * @inner
     * @constant
     * @type {string}
     * @default
     */
    var placeholderKey = "full-screen-controller-placeholder";

    /**
     * Get multiple CSS properties.
     *
     * @inner
     * @param {jQuery} element An element.
     * @param {string[]} propertyNames The names of the CSS properties to get.
     * @return {object} The named CSS properties on the element.
     */
    function getCSSProperties(element, propertyNames) {
        return _.reduce(propertyNames, function (properties, propertyName) {
            properties[propertyName] = element.css(propertyName);
            return properties;
        }, {});
    }

    /**
     * Create and insert a placeholder for an element.
     *
     * Placeholders are used to prevent layout jumps when elements are removed
     * from the flow. A reference to the placeholder is stored on the element.
     *
     * @inner
     * @param {jQuery} element The element.
     */
    function insertPlaceholder(element) {
        var placeholder;

        placeholder = jQuery("<div>").css({
            height: element.outerHeight()
        }).insertAfter(element);

        element.data(placeholderKey, placeholder);
    }

    /**
     * Remove an element's placeholder.
     *
     * @inner
     * @param {jQuery} element The element.
     */
    function removePlaceholder(element) {
        element.data(placeholderKey).remove();
        element.removeData(placeholderKey);
    }

    /**
     * Restore an element's original CSS properties.
     *
     * @inner
     * @param {jQuery} element An element.
     */
    function restoreCSSProperties(element) {
        var originalProperties = element.data(originalPropertiesKey);
        element.css(originalProperties).removeData(originalPropertiesKey);
    }

    /**
     * Set CSS properties on an element.
     *
     * The original values are stored on the element so they can be restored.
     *
     * @inner
     * @param {jQuery} element An element.
     * @param {object} properties The CSS properties to set.
     */
    function setCSSProperties(element, properties) {
        var originalProperties = getCSSProperties(element, _.keys(properties));
        element.css(properties).data(originalPropertiesKey, originalProperties);
    }

    AJS.namespace("JIRA.WorkflowDesigner.FullScreenController");

    /**
     * Methods for making elements occupy the whole window.
     *
     * @namespace
     */
    JIRA.WorkflowDesigner.FullScreenController = {
        /**
         * Make an element occupy the whole window.
         *
         * @param {element} element The element.
         * @return {jQuery.Deferred} A deferred that is resolved on completion.
         */
        enterFullScreen: function (element) {
            var body = jQuery("body"),
                deferred = jQuery.Deferred(),
                document = jQuery(window.document);

            element = jQuery(element);
            insertPlaceholder(element);
            setCSSProperties(element, {
                height: element.outerHeight(),
                left: element.offset().left - document.scrollLeft(),
                position: "fixed",
                top: element.offset().top - document.scrollTop(),
                width: element.outerWidth(),
                zIndex: 2000
            });

            setCSSProperties(body, {
                overflow: "hidden"
            });

            element.appendTo(body);
            element.animate({
                height: "100%",
                left: 0,
                top: 0,
                width: "100%"
            }, animationDuration, deferred.resolve);

            return deferred.promise();
        },

        /**
         * Restore an element to its original position/size.
         *
         * @param {element} element The element.
         * @return {jQuery.Deferred} A deferred that is resolved on completion.
         */
        exitFullScreen: function (element) {
            var body = jQuery("body"),
                deferred = jQuery.Deferred(),
                document = jQuery(window.document),
                placeholder;

            function restoreElement() {
                element.insertBefore(placeholder);
                removePlaceholder(element);
                restoreCSSProperties(element);
                deferred.resolve();
            }

            element = jQuery(element);
            placeholder = element.data(placeholderKey);

            restoreCSSProperties(body);
            element.animate({
                height: placeholder.outerHeight(),
                left: placeholder.offset().left - document.scrollLeft(),
                top: placeholder.offset().top - document.scrollTop(),
                width: placeholder.outerWidth()
            }, animationDuration, restoreElement);

            return deferred.promise();
        }
    };
}());