(function () {
    AJS.namespace("JIRA.WorkflowDesigner.SVGUtilities");

    /**
     * SVG utility methods.
     *
     * @namespace
     */
    JIRA.WorkflowDesigner.SVGUtilities = {
        /**
         * Convert a point from screen space to SVG space.
         *
         * The given point should be relative to the document's frame,
         * not its root element (i.e. scrolling shouldn't change it).
         *
         * @param {element} element An SVG element.
         * @param {number} x The point's x coordinate in screen space.
         * @param {number} y The point's y coordinate in screen space.
         */
        fromScreenToSVGCoordinate: function (element, x, y) {
            element = jQuery(element).get(0);
            return transformPoint(element, element.getScreenCTM().inverse(), x, y);
        },

        /**
         * Convert a point from SVG space to page space.
         *
         * The returned point is relative to the document's root element.
         *
         * @param {element} element An SVG element.
         * @param {number} x The point's x coordinate in SVG space.
         * @param {number} y The point's y coordinate in SVG space.
         * @return {object} The point in page space.
         */
        fromSVGToPageCoordinate: function (element, x, y) {
            var document = jQuery(window.document),
                point = JIRA.WorkflowDesigner.SVGUtilities.fromSVGToScreenCoordinate(element, x, y);

            point.x += document.scrollLeft();
            point.y += document.scrollTop();
            return point;
        },

        /**
         * Convert a point from SVG space to screen space.
         *
         * The returned point is relative to the document's frame,
         * not its root element (i.e. scrolling won't change it).
         *
         * @param {element} element An SVG element.
         * @param {number} x The point's x coordinate in SVG space.
         * @param {number} y The point's y coordinate in SVG space.
         * @return {object} The point in screen space.
         */
        fromSVGToScreenCoordinate: function (element, x, y) {
            element = jQuery(element).get(0);
            return transformPoint(element, element.getScreenCTM(), x, y);
        },

        /**
         * Calculate an SVG element's bounding box in page space.
         *
         * @param {element} element An SVG element.
         * @return {object} <tt>element</tt>'s bounding box in page space.
         */
        getPageBoundingBox: function (element) {
            var bottomRight,
                boundingBox = jQuery(element).get(0).getBBox(),
                topLeft;

            topLeft = JIRA.WorkflowDesigner.SVGUtilities.fromSVGToPageCoordinate(element, boundingBox.x, boundingBox.y);
            bottomRight = JIRA.WorkflowDesigner.SVGUtilities.fromSVGToPageCoordinate(
                    element,
                    boundingBox.x + boundingBox.width,
                    boundingBox.y + boundingBox.height
            );

            return {
                height: bottomRight.y - topLeft.y,
                width: bottomRight.x - topLeft.x,
                x: topLeft.x,
                y: topLeft.y
            };
        }
    };

    /**
     * Apply a matrix transformation to a point.
     *
     * @inner
     * @param {element} element An SVG element.
     * @param {SVGMatrix} matrix The matrix to apply.
     * @param {number} x The point's x coordinate.
     * @param {number} y The point's y coordinate.
     * @return {object} The result of applying <tt>matrix</tt> to the point.
     */
    function transformPoint(element, matrix, x, y) {
        var point = jQuery(element).closest("svg").get(0).createSVGPoint();
        point.x = x;
        point.y = y;

        point = point.matrixTransform(matrix);
        return {x: point.x, y: point.y};
    }
}());