(function () {
    var addedFirstStatus, addedFirstTransition;

    /**
     * Triggers a workflow analytic event with the passed name and properties.
     *
     * @inner
     * @param {string} name Event name.
     * @param {object} properties Event properties.
     */
    function triggerEvent(name, properties) {
        if (AJS.EventQueue) {
            AJS.EventQueue.push({
                name: "administration.workflow." + name,
                properties: properties
            });
        }
    }

    /**
     * Attaches a click handler that will trigger the specified analytics event
     * when an element matching the specified selector is clicked.
     *
     * @inner
     * @param {string} selector jQuery selector of the element to match.
     * @param {string} eventName Name of the event to trigger when the matching element is clicked.
     */
    function attachOnClickEventTrigger(selector, eventName) {
        jQuery(document).on("click", selector, function() {
            triggerEvent(eventName);
        });
    }

    attachOnClickEventTrigger(".enter-full-screen", "fullscreen");
    attachOnClickEventTrigger(".transition-properties", "view.transition.properties");
    attachOnClickEventTrigger(".transition-conditions", "view.transition.conditions");
    attachOnClickEventTrigger(".transition-validators", "view.transition.validators");
    attachOnClickEventTrigger(".transition-postfunctions", "view.transition.postfunctions");
    attachOnClickEventTrigger(".status-properties", "view.status.properties");

    /**
     * Triggers the Analytics event 'administration.workflow.edit' with the properties:
     *
     * * mode: 'new-diagram'
     * * action: <action param>
     * * object: <object param>
     *
     * @inner
     * @param {string} action Value to use in the 'action' property
     * @param {string} object Value to use in the 'object' property
     */
    function triggerEditEvent(action, object) {
        // This could have been handled by JIRA's analytics.js, using the custom event
        // "wfd-edit-action". However, that method always uses properties.mode="diagram"
        // so we need to add the event to the EventQueue ourselves.
        triggerEvent("edit", {
            mode: "new-diagram",
            action: action,
            object: object
        });
    }

    AJS.namespace("JIRA.WorkflowDesigner.Analytics");

    /**
     * @namespace JIRA.WorkflowDesigner.Analytics
     */
    JIRA.WorkflowDesigner.Analytics = {
        /**
         * Reset the analytics object, exposed for testing.
         *
         * @private
         */
        _reset: function () {
            addedFirstStatus = false;
            addedFirstTransition = false;
        },

        /**
         * Triggers an event for toggling label visibility.
         *
         * @param {boolean} show Whether the labels were shown or hidden.
         */
        triggerToggleLabels: function(show) {
            triggerEvent("toggle.labels." + (show ? "on" : "off"));
        },

        /**
         * Triggers an event for adding a status
         */
        triggerAddStatus: function () {
            triggerEditEvent("add", "status");
        },

        /**
         * Triggers an event for updating a status
         */
        triggerUpdateStatus: function () {
            triggerEditEvent("update", "status");
        },

        /**
         * Triggers an event for adding a step
         */
        triggerAddStep: function () {
            triggerEditEvent("add", "step");
        },

        /**
         * Trigger an event for the very first time a user adds a status.
         *
         * @param {number} duration The time it took the user to add a status (in milliseconds).
         */
        triggerFirstAddStatus: function (duration) {
            var propertyKey = "jira.workflowdesigner.added.first.status",
                request = jQuery.Deferred(),
                UserPropertiesAJAXManager = JIRA.WorkflowDesigner.IO.AJAX.UserPropertiesAJAXManager;

            addedFirstStatus || (request = UserPropertiesAJAXManager.getProperty(propertyKey));
            request.done(function (property) {
                var shouldTrigger = !property.value;
                if (shouldTrigger) {
                    triggerEvent("first.add.status", {duration: duration});
                    UserPropertiesAJAXManager.setProperty(propertyKey, true);
                }

                addedFirstStatus = true;
            });
        },

        /**
         * Triggers an event for updating a step
         */
        triggerUpdateStep: function () {
            triggerEditEvent("update", "step");
        },

        /**
         * Triggers an event for removing a step
         */
        triggerRemoveStep: function () {
            triggerEditEvent("remove", "step");
        },

        triggerAddCommonTransition: function () {
            triggerEditEvent("add", "commontransition");
        },

        /**
         * Triggers an event for adding a transition
         */
        triggerAddTransition: function () {
            triggerEditEvent("add", "transition");
        },

        /**
         * Trigger an event for the very first time a user adds a transition.
         *
         * @param {number} duration The time it took the user to add a transition (in milliseconds).
         */
        triggerFirstAddTransition: function (duration) {
            var propertyKey = "jira.workflowdesigner.added.first.transition",
                request = jQuery.Deferred(),
                UserPropertiesAJAXManager = JIRA.WorkflowDesigner.IO.AJAX.UserPropertiesAJAXManager;

            addedFirstTransition || (request = UserPropertiesAJAXManager.getProperty(propertyKey));
            request.done(function (property) {
                var shouldTrigger = !property.value;
                if (shouldTrigger) {
                    triggerEvent("first.add.transition", {duration: duration});
                    UserPropertiesAJAXManager.setProperty(propertyKey, true);
                }

                addedFirstTransition = true;
            });
        },

        /**
         * Triggers an event for updating a transition
         */
        triggerUpdateTransition: function () {
            triggerEditEvent("update", "transition");
        },

        /**
         * Triggers an event for removing a transition
         */
        triggerRemoveTransition: function () {
            triggerEditEvent("remove", "transition");
        },

        /**
         * Triggers an event for adding a global transition
         */
        triggerAddGlobalTransition: function () {
            triggerEditEvent("add", "globaltransition");
        },

        /**
         * Triggers an event for updating a global transition
         */
        triggerUpdateGlobalTransition: function () {
            triggerEditEvent("update", "globaltransition");
        },

        /**
         * Triggers an event for removing a global transition
         */
        triggerRemoveGlobalTransition: function () {
            triggerEditEvent("remove", "globaltransition");
        }
    };
}());