(function () {
    /**
     * Shortcut to JIRA.WorkflowDesigner.Policy.Canvas.SnapToGeometry.SnapPoints
     *
     * @inner
     * @type {function}
     */
    var SnapPoints;

    AJS.namespace("JIRA.WorkflowDesigner.Policy.Canvas.SnapToGeometry.GuideLineEditPolicy");

    JIRA.WorkflowDesigner.Policy.Canvas.SnapToGeometry.GuideLineEditPolicy = draw2d.policy.canvas.CanvasPolicy.extend(
    /** @lends JIRA.WorkflowDesigner.Policy.Canvas.SnapToGeometry.GuideLineEditPolicy# */
    {
        /**
         * The colour of guide lines.
         *
         * @constant
         * @type {string}
         * @default
         */
        GUIDE_LINE_COLOUR: "#ff0000",

        /**
         * Initialise the instance.
         *
         * Store references to dependencies.
         *
         * @classdesc
         * An edit policy that draws guide lines between aligned figures.
         *
         * Only the figure that is currently being dragged is considered.
         * @constructs
         * @extends draw2d.policy.canvas.CanvasPolicy
         */
        init: function () {
            this._super.apply(this, arguments);

            // This can't be done earlier as the resources wouldn't have loaded.
            SnapPoints = JIRA.WorkflowDesigner.Policy.Canvas.SnapToGeometry.SnapPoints;
        },

        /**
         * Collect all figures that are participating in snap to geometry.
         *
         * @private
         */
        _collectFigures: function () {
            var instance = this;

            // They've already been collected.
            if (this.figures) {
                return;
            }

            this.figures = [];
            this.canvas.getFigures().each(function (index, figure) {
                if (figure.snapToGeometry) {
                    if (figure.isInDragDrop) {
                        instance.dragFigure = figure;
                    } else {
                        instance.figures.push(figure);
                    }
                }
            });
        },

        /**
         * Collect all snap points on the canvas.
         *
         * @private
         */
        _collectSnapPoints: function () {
            // They've already been collected.
            if (this.SnapPoints) {
                return;
            }

            this.SnapPoints = {
                x: SnapPoints.getXSnapPoints(this.figures),
                y: SnapPoints.getYSnapPoints(this.figures)
            };
        },

        /**
         * Draw a guide line on the canvas.
         *
         * @param {number} x1 The x coordinate of the line's start point.
         * @param {number} y1 The y coordinate of the line's start point.
         * @param {number} x2 The x coordinate of the line's end point.
         * @param {number} y2 The y coordinate of the line's end point.
         * @private
         */
        _drawGuideLine: function (x1, y1, x2, y2) {
            var path = [
                "M", x1, y1,
                "L", x2, y2
            ].join(" ");

            this.guideLines || (this.guideLines = []);
            this.guideLines.push(this.canvas.paper.path(path).attr({
                stroke: this.GUIDE_LINE_COLOUR
            }));
        },

        /**
         * Draw guide lines between two sets of snap points.
         *
         * @param {function} draw A function to draw a guide line.
         * @param {object[]} sourcePoints The first set of snap points.
         * @param {object[]} targetPoints The second set of snap points.
         * @private
         */
        _drawGuideLines: function (draw, sourcePoints, targetPoints) {
            var pairs = _.cartesianProduct(sourcePoints, targetPoints);

            _.each(pairs, function (pair) {
                if (this._equalSnapPoints(pair[0], pair[1])) {
                    draw.apply(this, pair);
                }
            }, this);
        },

        /**
         * Draw guide lines between figures that are aligned vertically.
         *
         * @private
         */
        _drawHorizontalGuideLines: function () {
            var snapPoints;

            function draw(sourcePoint, targetPoint) {
                var maximum = Math.max(sourcePoint.rectangle.getRight(), targetPoint.rectangle.getRight()),
                    minimum = Math.min(sourcePoint.rectangle.getLeft(), targetPoint.rectangle.getLeft());

                this._drawGuideLine(
                    minimum, sourcePoint.value,
                    maximum, sourcePoint.value
                );
            }

            snapPoints = SnapPoints.getYSnapPoints(this.dragFigure);
            this._drawGuideLines(draw, snapPoints, this.SnapPoints.y);
        },

        /**
         * Draw guide lines between figures that are aligned horizontally.
         *
         * @private
         */
        _drawVerticalGuideLines: function () {
            var snapPoints;

            function draw(sourcePoint, targetPoint) {
                var maximum = Math.max(sourcePoint.rectangle.getBottom(), targetPoint.rectangle.getBottom()),
                    minimum = Math.min(sourcePoint.rectangle.getTop(), targetPoint.rectangle.getTop());

                this._drawGuideLine(
                    sourcePoint.value, minimum,
                    sourcePoint.value, maximum
                );
            }

            snapPoints = SnapPoints.getXSnapPoints(this.dragFigure);
            this._drawGuideLines(draw, snapPoints, this.SnapPoints.x);
        },

        /**
         * Determine if two snap points are equal.
         *
         * @param {object} a The first snap point.
         * @param {object} b The second snap point.
         * @return {boolean} Whether the two snap points are equal.
         * @private
         */
        _equalSnapPoints: function (a, b) {
            return Math.abs(a.value - b.value) <= 0.1;
        },

        /**
         * @param {draw2d.Canvas} canvas Canvas where this policy is being installed
         */
        onInstall: function (canvas) {
            this.canvas = canvas;
        },

        /**
         * @method
         */
        onMouseDrag: function () {
            this._collectFigures();
            this._removeGuideLines();

            if (!this.dragFigure || this.figures.length === 0) {
                return;
            }

            this._collectSnapPoints();
            this._drawHorizontalGuideLines();
            this._drawVerticalGuideLines();
        },

        /**
         * Delete cached data.
         */
        onMouseUp: function () {
            delete this.dragFigure;
            delete this.figures;
            delete this.SnapPoints;
            this._removeGuideLines();
        },

        /**
         * @method
         */
        onUninstall: function () {
            delete this.canvas;
        },

        /**
         * Remove all guide lines from the canvas.
         *
         * @private
         */
        _removeGuideLines: function () {
            _.invoke(this.guideLines, "remove");
            this.guideLines = [];
        }
    });
}());