define('jira/project-templates/add-project-controller-impl', [
    'jira/project-templates/add-project-view',
    'jira/project-templates/project-key-generator',
    'jira/project-templates/dialog-controller',
    'jira/project-templates/config',
    'jira/lib/class',
    'jquery',
    'underscore'
], function (
    AddProjectView,
    ProjectKeyGenerator,
    DialogController,
    Config,
    Class,
    $,
    _
){
    var contextPath = AJS.contextPath();
    var trigger = AJS.trigger;

    return Class.extend({
        postProjectCreationCallbacks: [],
        projectKeyValidationCallbacks: [],
        projectNamesUpperCased: [],

        init: function (options) {
            this._window = options.window || window; // used for mocking 'window' in QUnit
            _.bindAll(this, "projectCreatedHandler", "projectValidationFailedHandler", "nameTimeout", "keyTimeout", "submit");
        },

        initCreateShared: function (existingProjectData) {
            this.existingProjectData = existingProjectData;
            this._initAddProject(AJS.I18n.getText('add.project.dialog.create.shared'));
        },

        initProjectTemplate: function (webItemData) {
            this.existingProjectData = undefined;
            this.webItemData = webItemData;
            this._initAddProject(webItemData.name);
        },

        _initAddProject: function (title) {
            this.desiredKeyLength = 4;
            this.lastKeyValidated = "";

            this._getExistingProjects();
            ProjectKeyGenerator.init(this.desiredKeyLength, Config.model.maxKeyLength);

            AddProjectView.prepareDialog(title);
            AddProjectView.draw({
                webItemData: this.webItemData || {},
                maxKeyLength: Config.model.maxKeyLength,
                maxNameLength: Config.model.maxNameLength,
                currentUserName: Config.model.currentUserName,
                currentUserDisplayName: Config.model.currentUserDisplayName,
                currentUserAvatarUrl: Config.model.currentUserAvatarUrl,
                errors: {}
            });
        },

        _resetProjectLeadValues: function () {
            Config.model.currentUserDisplayName = "";
            Config.model.currentUserName = "";
            Config.model.currentUserAvatarUrl = "";
        },

        _hasFullyConfiguredProjectLead: function () {
            return AddProjectView.getLeadDisplayName() && AddProjectView.getAvatarUrlOfSelectedLead() != "";
        },

        submit: function (event) {
            // This data is retrieved before the input fields are disabled (in enterLoadingState()) to avoid null-pointer exceptions when the fields are empty.
            var formData;
            if (this.existingProjectData) {
                formData = {
                    "key": AddProjectView.getKey(),
                    "name": AddProjectView.getName(),
                    "lead": AddProjectView.getLeadUserName() && AddProjectView.getLeadUserName()[0]
                };
            } else {
                formData = jQuery.param(AddProjectView.getAddProjectForm().serializeArray().map(function(param) {
                    // upper case is "visually" enforced by text-transform: uppercase
                    if (param.name === "key") {
                        param.value = param.value.toUpperCase();
                    }
                    return param
                }));
            }

            if (AddProjectView.hasInlineErrors()) {
                return;
            }

            if (AddProjectView.enterLoadingState()) {
                // store lead so we can repopulate the user picker for when the submit fails
                if (this._hasFullyConfiguredProjectLead()) {
                    Config.model.currentUserDisplayName = AddProjectView.getLeadDisplayName();
                    Config.model.currentUserName = Config.model.currentUserDisplayName ? AddProjectView.getLeadUserName() : "";
                    Config.model.currentUserAvatarUrl = AddProjectView.getAvatarUrlOfSelectedLead();
                } else {
                    this._resetProjectLeadValues();
                }

                if (this.existingProjectData) {
                    $.ajax({
                        url: contextPath + "/rest/project-templates/1.0/createshared/" + this.existingProjectData.existingProjectId,
                        type: "POST",
                        contentType: "application/json",
                        data: JSON.stringify(formData)
                    })
                        .done(this.projectCreatedHandler)
                        .fail(this.projectValidationFailedHandler);
                } else {
                    $.ajax({
                        url: contextPath +
                            (this.webItemData.demoProject ?
                                "/rest/jira-importers-plugin/1.0/demo/create" :
                                "/rest/project-templates/1.0/templates"),
                        type: "POST",
                        data: formData,
                        headers: {
                            "X-Atlassian-Token": "no-check"
                        }
                    })
                        .done(this.projectCreatedHandler)
                        .fail(this.projectValidationFailedHandler);
                }
            }
        },

        registerPostProjectCreationCallback: function (callback) {
            this.postProjectCreationCallbacks.push(callback);
        },

        registerProjectKeyValidationCallback: function (callback) {
            this.projectKeyValidationCallbacks.push(callback);
        },

        localStoragePrefix: "jira.projecttemplates.",

        projectCreatedHandler: function (data) {
            if (this.existingProjectData) {
                trigger('analyticsEvent', {name: 'jira.project.templates.dialog.create.shared.project.create.success'});
            } else if (this.webItemData.demoProject) {
                trigger('analyticsEvent', {name: 'jira.project.templates.dialog.create.demo.success'});
            } else {
                trigger('analyticsEvent', {name: 'jira.project.templates.dialog.create.project.success'});
            }
            AddProjectView.avoidDirtyFormWarning();

            // Put the created projects into local storage
            var localStoragePrefix = this.localStoragePrefix;
            var remoteProjectTypes = ["confluenceProject", "fisheyeProject", "crucibleProject", "bambooProject"];
            _.map(remoteProjectTypes, function (remoteProject) {
                if (data.remoteProjectLinks && data.remoteProjectLinks[remoteProject]) {
                    localStorage.setItem(localStoragePrefix + remoteProject, data.remoteProjectLinks[remoteProject]);
                }
            });


            // If there are any post Project Created callbacks registered, call these. Otherwise redirect to the desired return URL.
            if (!_.isEmpty(this.postProjectCreationCallbacks)) {
                DialogController.dialog.addPage("post-project-created-page");
                _.each(this.postProjectCreationCallbacks, function (callback) {
                    callback(DialogController.dialog, data.returnUrl, data.projectId, data.projectKey, data.projectName);
                });
            } else {
                this._window.location = contextPath + data.returnUrl;
            }
        },

        projectValidationFailedHandler: function (jqXhr, textStatus) {
            if (this.existingProjectData) {
                trigger('analyticsEvent', {name: 'jira.project.templates.dialog.create.shared.project.create.failure'});
            } else if (this.webItemData.demoProject) {
                trigger('analyticsEvent', {name: 'jira.project.templates.dialog.create.demo.failure'});
            } else {
                trigger('analyticsEvent', {name: 'jira.project.templates.dialog.create.project.failure'});
            }

            var errors = {};
            if (this.isBadRequest(jqXhr)) {
                errors = JSON.parse(jqXhr.responseText);
            } else if (this.isUnDefinedServerSideError(jqXhr)) {
                errors = {"errorMessages": [AJS.I18n.getText("add.projects.templates.error.undefined.server.side.error")]}
            } else if (this.isDefinedServerSideError(jqXhr)) {
                var errorResponse = JSON.parse(jqXhr.responseText);
                // Project template configuration threw an exception: Project has been created but the template hasn't been configured
                JIRA.Messages.showReloadErrorMsg(errorResponse.message);
                AddProjectView.avoidDirtyFormWarning();
                this._window.location = contextPath + errorResponse.returnUrl;
                // Remain in loading state while refreshing.
                return;
            } else if (this.isTimeoutError(textStatus)) {
                errors = {"errorMessages": [AJS.I18n.getText("add.projects.templates.error.timeout")]}
            } else {
                // Different kind of error. Let's generate an errors object with a general error that contains the right information
                errors = {"errorMessages": [AJS.I18n.getText("add.projects.templates.error.unknownError", jqXhr.responseText)]}
            }

            AddProjectView.draw({
                webItemData: this.webItemData || {},
                errors: errors,
                currentName: AddProjectView.getName(),
                currentKey: AddProjectView.getKey(),
                currentUserDisplayName: Config.model.currentUserDisplayName,
                currentUserName: Config.model.currentUserName,
                currentUserAvatarUrl: Config.model.currentUserAvatarUrl
            });

            AddProjectView.hideLoadingState();
        },

        isBadRequest: function(jqXhr) {
            return (jqXhr.status === 400) ;
        },

        isUnDefinedServerSideError: function(jqXhr){
            if (jqXhr.status === 500) {
                try {
                    JSON.parse(jqXhr.responseText);
                } catch(e) {
                    return true;
                }
            }
            return false;
        },

        isDefinedServerSideError: function(jqXhr) {
            return jqXhr.status === 500 && !_.isUndefined(JSON.parse(jqXhr.responseText).message) ;
        },

        isTimeoutError: function(textStatus) {
            return textStatus === "timeout" ;
        },

        _updateAndValidateKey: function updateKey(key) {
            AddProjectView.setKey(key);
            this.validateKey();
        },

        _shouldUpdateKey: function _shouldUpdateKey() {
            return (AddProjectView.getKeyEdited() != "true");
        },

        autofillKeyIfNeeded: function autofillKeyIfNeeded() {
            if (this._shouldUpdateKey()) {
                var key = ProjectKeyGenerator.generate(AddProjectView.getName());
                // JRADEV-10797 - Rather than validate the key,
                // we'll pretend that a key is always invalid if it's less than 1 character long.
                if (key.length > 1) {
                    this._updateAndValidateKey(key);
                } else {
                    // Blank the key without validation.
                    AddProjectView.setKey("");
                }
            }
        },

        _doesProjectNameExists: function (name) {
            var x;
            for (x in this.projectNamesUpperCased) {
                if (name.toUpperCase() == this.projectNamesUpperCased[x]) {
                    return true;
                }
            }

            return false;
        },

        validateName: function () {
            var name = $.trim(AddProjectView.getName());

            if (!name) {
                return;
            }

            if (name.length < Config.model.minNameLength) {
                AddProjectView.showInlineErrorForName(AJS.I18n.getText("admin.errors.project.name.too.short", Config.model.minNameLength));
                return;
            }

            if (name.length > Config.model.maxNameLength) {
                AddProjectView.showInlineErrorForName(AJS.I18n.getText("admin.errors.project.name.too.long", Config.model.maxNameLength));
                return;
            }

            if (this._doesProjectNameExists(name)) {
                AddProjectView.showInlineErrorForName(AJS.I18n.getText("admin.errors.project.with.that.name.already.exists"));
                return;
            }

            AddProjectView.hideInlineErrorForName();
        },

        _performKeyValidationChecks: function (key) {
            var validationChecksDeferred = $.ajax({
                url: contextPath + "/rest/api/latest/projectvalidate/key?key=" + key.toUpperCase()
            });

            validationChecksDeferred.done(_.bind(function (errors) {
                if (errors.errors && errors.errors["projectKey"]) {
                    AddProjectView.showInlineErrorForKey(errors.errors["projectKey"]);
                } else {
                    var foundError = false;
                    _.each(this.projectKeyValidationCallbacks, function (callback) {
                        var errors = callback(key.toUpperCase());
                        if (errors.errors && errors.errors["projectKey"]) {
                            foundError = true;
                            AddProjectView.showInlineErrorForKey(errors.errors["projectKey"]);
                        }
                    });
                    if (!foundError) {
                        AddProjectView.hideInlineErrorForKey();
                    }
                }
            }, this));
        },

        validateKey: function validateKey() {
            var key = AddProjectView.getKey();

            // Only validate the key if it has changed since the last time we validated it
            if (this.lastKeyValidated === key) {
                return;
            }

            if (key) {
                this.lastKeyValidated = key;
                this._performKeyValidationChecks(key);
            } else {
                AddProjectView.hideInlineErrorForKey();
            }
        },

        nameTimeout: function nameTimeout() {
            this.autofillKeyIfNeeded();
        },

        keyTimeout: function keyTimeout() {
            AddProjectView.setKeyEdited();
        },

        _getExistingProjects: function () {
            // Avoid retrieving the list twice (e.g. after hitting the 'Back' button and selecting new template)
            if (this.projectNamesUpperCased.length > 0) {
                return this.projectNamesUpperCased;
            }

            var existingProjectsDeferred = $.ajax({
                url: contextPath + "/rest/api/latest/project"
            });

            existingProjectsDeferred.done(_.bind(function (projects) {
                this.projectNamesUpperCased = _.map(projects, function (project) {
                    return project.name.toUpperCase();
                });
            }, this));
        }
    });
});
define('jira/project-templates/add-project-controller', [
    'jira/project-templates/add-project-controller-impl',
    'jquery'
], function(
    AddProjectControllerImpl,
    $
){
    return new AddProjectControllerImpl({el: $(document)});
});
AJS.namespace('JPT.AddProjectController', null, require('jira/project-templates/add-project-controller'));
