define("jira-github-importer/fetchData", [
    "jquery",
    "./lib/contextPath",
    "./lib/log",
    "./lib/window"
], function (
    $,
    contextPath,
    log,
    window
) {
    "use strict";

    /**
     * Perform some human-friendly formatting of progress of an import.
     *
     * @param {number} currentIssue
     * @param {number} [totalIssues=1000]
     * @returns {{percentage: (number|*), message: string}}
     */
    function getProgress(currentIssue, totalIssues) {
        var percentage;
        if (totalIssues !== null) {
            percentage = (currentIssue / totalIssues) * 100;
        } else {
            percentage = (currentIssue / 10000) * 100;
        }
        percentage = Math.round(percentage);
        var message = currentIssue + "/" + (totalIssues !== null ? totalIssues : "?");
        return {percentage: percentage, message: message};
    }

    /**
     * Update the UI with the progress of the fetch.
     *
     * @param {string} fetchProgress.error
     * @param {string} fetchProgress.warning
     * @param {number} fetchProgress.currentClosedIssue
     * @param {number} fetchProgress.totalClosedIssues
     * @param {number} fetchProgress.currentOpenIssue
     * @param {number} fetchProgress.totalOpenIssues
     * @param {number} fetchProgress.currentProject
     * @param {number} fetchProgress.totalProjects
     * @param {boolean} fetchProgress.finished
     */
    function notifyProgress(fetchProgress) {
        var progress = getProgress(fetchProgress.currentClosedIssue, fetchProgress.totalClosedIssues);

        updateMessage("error", fetchProgress.error);
        updateMessage("warning", fetchProgress.warning);

        $("#closedIssuesProgressbar").progressBar(progress.percentage);
        $("#closedStatus").text(progress.message);

        progress = getProgress(fetchProgress.currentOpenIssue, fetchProgress.totalOpenIssues);
        $("#openIssuesProgressbar").progressBar(progress.percentage);
        $("#openStatus").text(progress.message);

        progress = getProgress(fetchProgress.currentProject, fetchProgress.totalProjects);
        $("#projectsProgressBar").progressBar(progress.percentage);
        $("#projectStatus").text(progress.message);
    }

    /**
     * Update the message
     *
     * @param {"error"|"warning"} type
     * @param {string} text
     */
    function updateMessage(type, text) {
        if (text === null) {
            $("#" + type + "Box").addClass("hidden");
        } else {
            $("#" + type + "Message").text(text);
            $("#" + type + "Box").removeClass("hidden");
        }
    }

    /**
     * Backoff the polling delay.
     *
     * @param {number} delay
     * @returns {number}
     */
    function extendedPollDelay(delay) {
        if (delay < 3000) {
            delay *= 1.2;
        }

        return delay;
    }

    return {
        /**
         * Poll for progress of the import and update the user with the status. Repeat this process until the import
         * has completed.
         *
         * @param {string} externalSystem The external system to poll.
         * @param {number} [nextPollDelay=500] The number of milliseconds until the subsequent poll.
         */
        pollProgress: function (externalSystem, nextPollDelay) {
            var that = this;
            nextPollDelay = typeof nextPollDelay == "undefined" ? 500 : nextPollDelay;

            log("Polling progress");
            $.ajax({
                url: contextPath + "/rest/github-importer-plugin/1.0/fetchData/" + externalSystem + "/progress",
                type: "GET",
                cache: false
            }).done(function (progress) {
                notifyProgress(progress);

                if (progress.finished === true) {
                    window.document.location.href = contextPath + "/secure/admin/GithubLabelMappingPage!default.jspa?externalSystem=" + externalSystem;
                } else {
                    window.setTimeout(function () {
                        // gradually increase the polling delay
                        that.pollProgress(externalSystem, extendedPollDelay(nextPollDelay))
                    }, nextPollDelay);
                }
            }).fail(function (jqXHR, textStatus, errorThrown) {
                log(errorThrown);
                updateMessage("error", "Unable to check fetch progress (" + errorThrown + ")");
                window.setTimeout(function () {
                    that.pollProgress(externalSystem);
                }, 2000);
            });
        },

        /**
         * Cancel the fetching of data.
         *
         * @param {string} externalSystem The external system which is being fetched.
         */
        cancel: function (externalSystem) {
            log("Canceling fetch");
            $.ajax({
                url: contextPath + "/rest/github-importer-plugin/1.0/fetchData/" + externalSystem + "/cancel",
                type: "POST",
                cache: false
            }).done(function () {
                window.document.location.href = contextPath + "/GithubProjectMappingPage!default.jspa?externalSystem=" + externalSystem;
            }).fail(function (jqXHR, textStatus, errorThrown) {
                log(errorThrown);
                updateMessage("error", "Unable to cancel fetch (" + errorThrown + ")");
            });

            $("#cancelBtn").attr('disabled', 'disabled');
        }
    };
});
