package com.atlassian.jira.plugins.importer.github.web;

import com.atlassian.gzipfilter.org.apache.commons.lang.StringUtils;
import com.atlassian.jira.component.ComponentAccessor;
import com.atlassian.jira.issue.fields.config.manager.IssueTypeSchemeManager;
import com.atlassian.jira.plugins.importer.github.config.ConfigBean;
import com.atlassian.jira.plugins.importer.github.config.SchemeStatusMapping;
import com.atlassian.jira.plugins.importer.github.rest.workflow.WorkflowResource;
import com.atlassian.jira.plugins.importer.github.rest.workflow.WorkflowService;
import com.atlassian.jira.plugins.importer.tracking.UsageTrackingService;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.project.ProjectManager;
import com.atlassian.jira.security.xsrf.RequiresXsrfCheck;
import com.atlassian.jira.workflow.AssignableWorkflowScheme;
import com.atlassian.jira.workflow.JiraWorkflow;
import com.atlassian.jira.workflow.WorkflowScheme;
import com.atlassian.jira.workflow.WorkflowSchemeManager;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.plugin.web.WebInterfaceManager;
import com.google.common.base.Function;
import com.google.common.base.Optional;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import org.codehaus.jackson.map.ObjectMapper;
import webwork.action.ActionContext;

import javax.annotation.Nullable;
import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Allows the user select a workflow scheme and map the GitHub issue state to appropriate workflow statuses.
 */
public class GithubWorkflowPage extends GithubImportProcessSupport {

    public static final String SCHEME_SELECT_ID = "workflowSchemeSelect";
    private final WorkflowService workflowService;


	public GithubWorkflowPage(@ComponentImport UsageTrackingService usageTrackingService,
							  @ComponentImport WebInterfaceManager webInterfaceManager,
							  @ComponentImport PluginAccessor pluginAccessor,
							  @ComponentImport ProjectManager projectManager,
							  @ComponentImport IssueTypeSchemeManager issueTypeSchemeManager,
							  @ComponentImport WorkflowSchemeManager workflowSchemeManager) {
		super(usageTrackingService, webInterfaceManager, pluginAccessor, projectManager, issueTypeSchemeManager, workflowSchemeManager);
		this.workflowService = new WorkflowService(ComponentAccessor.getWorkflowManager(), workflowSchemeManager, getConstantsManager());
	}

	public boolean displayLimitedWorkflowSchemesWarning() {
		return getWorkflowScheme().isPresent();
	}

	public String getWorkflowName() {
		final Optional<WorkflowScheme> workflowScheme = getWorkflowScheme();
		return workflowScheme.isPresent() ? workflowScheme.get().getName() : "???";
	}

	private Optional<WorkflowScheme> getWorkflowScheme() {
		final Iterable<WorkflowScheme> schemes = Iterables.transform(getExistingDestinationProjects(), new Function<Project, WorkflowScheme>() {
			@Override
			public WorkflowScheme apply(@Nullable Project input) {
				return workflowSchemeManager.getWorkflowSchemeObj(input);
			}
		});
		return Optional.fromNullable(Iterables.getFirst(Iterables.filter(schemes, Predicates.notNull()), null));
	}

	@Override
    @RequiresXsrfCheck
    protected String doExecute() throws Exception {
        return super.doExecute();
    }

    @Override
    protected void doValidation() {

		//even if user is pressing back - remember his settings!
		Map<String, String> actionParams = ActionContext.getSingleValueParameters();
		String workflowSchemeName = actionParams.get(SCHEME_SELECT_ID);
		if (StringUtils.isBlank(workflowSchemeName)) {
			addError(SCHEME_SELECT_ID, getText("com.atlassian.jira.plugins.importer.github.workflow.error.noScheme"));
			getConfigBean().setSchemeStatusMapping(null);
		} else {
			populateStatusMappings();
		}
		super.doValidation();
    }

    private void populateStatusMappings() {
        Map<String,String> actionParams = ActionContext.getSingleValueParameters();

        String workflowSchemeName = actionParams.get(SCHEME_SELECT_ID);
        if( WorkflowResource.DEFAULT_WORKFLOW_SCHEME_KEY.equals(workflowSchemeName) ) {
            workflowSchemeName = null;
        }

        Collection<JiraWorkflow> workflows = workflowService.getSchemeWorkflows(workflowSchemeName);

        SchemeStatusMapping schemeStatusMapping = new SchemeStatusMapping(workflowSchemeName);
        for( JiraWorkflow workflow : workflows ) {
            String openStatus = actionParams.get(workflow.getName() + "_open_status");
            String closedStatus = actionParams.get(workflow.getName() + "_closed_status");

            SchemeStatusMapping.JiraStatusMapping statusMapping = new SchemeStatusMapping.JiraStatusMapping(openStatus,closedStatus);
            schemeStatusMapping.getWorkflowIdToStatusMapping().put(workflow.getName(), statusMapping);
        }

        getConfigBean().setSchemeStatusMapping(schemeStatusMapping);
    }

    @Override
    public String getFormTitle() {
        return getText("com.atlassian.jira.plugins.importer.github.workflow.title");
    }

    public Map<String, String> getWorkflowSchemeValues() {
		final List<Project> existingDestinationProjects = getExistingDestinationProjects();
		if(existingDestinationProjects.size() > 0) {
			final String name = getSchemeName();
			return ImmutableMap.of(name, name);
		}

        Map<String,String> values = new HashMap<String, String>();
        values.put(WorkflowResource.DEFAULT_WORKFLOW_SCHEME_KEY, workflowService.getDefaultSchemeName());

        for( String schemeName : workflowService.getSchemeNames() ) {
            values.put(schemeName, schemeName);
        }
        return values;
    }

	private String getSchemeName() {
		final Project project = getExistingDestinationProjects().get(0);
		final AssignableWorkflowScheme workflowScheme = workflowSchemeManager.getWorkflowSchemeObj(project);
		return workflowScheme.getName();
	}

	public String getSelectedWorkflowSchemeKey() {
		SchemeStatusMapping schemeStatusMapping = getConfigBean().getSchemeStatusMapping();
		final Map<String, String> workflowSchemeValues = getWorkflowSchemeValues();
		String keyFromExistingMapping = null;
		if(schemeStatusMapping != null) {
			if( schemeStatusMapping.getWorkflowSchemeName() == null ) {
				keyFromExistingMapping = WorkflowResource.DEFAULT_WORKFLOW_SCHEME_KEY;
			} else {
				keyFromExistingMapping = schemeStatusMapping.getWorkflowSchemeName();
			}
		}

		if(workflowSchemeValues.containsKey(keyFromExistingMapping)) {
			return keyFromExistingMapping;
		}

		if(workflowSchemeValues.size() == 1) {
			return workflowSchemeValues.keySet().iterator().next();
		}

		return "";
    }

    public String getStatusMappingJSON() {
        Map<String, SchemeStatusMapping.JiraStatusMapping> workflowIdToStatusMapping;

        SchemeStatusMapping schemeStatusMapping = getConfigBean().getSchemeStatusMapping();
        if( schemeStatusMapping!= null) {
            workflowIdToStatusMapping = schemeStatusMapping.getWorkflowIdToStatusMapping();
        } else {
            workflowIdToStatusMapping = new HashMap<String, SchemeStatusMapping.JiraStatusMapping>();
        }

        try {
            return new ObjectMapper().writeValueAsString(workflowIdToStatusMapping);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public ConfigBean getConfigBean() {
        try {
            if (getController() == null || getController().getImportProcessBeanFromSession() == null) {
                throw new IllegalStateException("Importer not intialized");
            }
            ConfigBean configBean = (ConfigBean) getController().getImportProcessBeanFromSession().getConfigBean();
            return configBean;
        } catch (ClassCastException e) {
            throw new RuntimeException(e);
        }
    }

    public String getSchemeSelectId() {
        return SCHEME_SELECT_ID;
    }

}
