package com.atlassian.jira.plugins.importer.github.web;

import com.atlassian.jira.bc.project.ProjectService;
import com.atlassian.jira.issue.fields.config.FieldConfigScheme;
import com.atlassian.jira.issue.fields.config.manager.IssueTypeSchemeManager;
import com.atlassian.jira.plugins.importer.extensions.ImporterController;
import com.atlassian.jira.plugins.importer.github.GithubImportProcessBean;
import com.atlassian.jira.plugins.importer.github.config.ConfigBean;
import com.atlassian.jira.plugins.importer.managers.CreateProjectManager;
import com.atlassian.jira.plugins.importer.tracking.UsageTrackingService;
import com.atlassian.jira.plugins.importer.web.ImportProcessBean;
import com.atlassian.jira.plugins.importer.web.ImporterProjectMappingsPage;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.project.ProjectManager;
import com.atlassian.jira.security.xsrf.RequiresXsrfCheck;
import com.atlassian.jira.workflow.AssignableWorkflowScheme;
import com.atlassian.jira.workflow.WorkflowSchemeManager;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.plugin.web.WebInterfaceManager;
import com.google.common.base.Function;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;

import javax.annotation.Nullable;
import java.util.Collection;

public class GithubProjectMappingPage extends ImporterProjectMappingsPage {

	protected final ProjectManager projectManager;
	protected final IssueTypeSchemeManager issueTypeSchemeManager;
	protected final WorkflowSchemeManager workflowSchemeManager;

	public GithubProjectMappingPage(
			 @ComponentImport UsageTrackingService usageTrackingService,
			 @ComponentImport ProjectService projectService,
			 @ComponentImport WebInterfaceManager webInterfaceManager,
			 @ComponentImport PluginAccessor pluginAccessor,
			 @ComponentImport ProjectManager projectManager,
			 @ComponentImport CreateProjectManager createProjectManager,
			 @ComponentImport IssueTypeSchemeManager issueTypeSchemeManager,
			 @ComponentImport WorkflowSchemeManager workflowSchemeManager) {
		super(usageTrackingService, projectService, webInterfaceManager, pluginAccessor, projectManager, createProjectManager);
		this.projectManager = projectManager;
		this.issueTypeSchemeManager = issueTypeSchemeManager;
		this.workflowSchemeManager = workflowSchemeManager;
	}

	@Override
	@RequiresXsrfCheck
	protected void doValidation() {
		super.doValidation();
		final Collection<Project> existingDestinationProjects = getExistingDestinationProjects();

		if(isMoreThanOneIssueTypeScheme(existingDestinationProjects)) {
			addErrorMessage(getI18nHelper().getText("com.atlassian.jira.plugins.importer.github.step.projectmapping.issuetypescheme.error"));
		}
		if(isMoreThanOneWorkflowSchemes(existingDestinationProjects)) {
			addErrorMessage(getI18nHelper().getText("com.atlassian.jira.plugins.importer.github.step.projectmapping.workflowscheme.error"));
		}
	}

	protected boolean isMoreThanOneIssueTypeScheme (Iterable<Project> projects) {
		final Iterable<Long> issueTypeSchemeIds = Iterables.transform(projects, new Function<Project, Long>() {
			@Override
			public Long apply(Project input) {
				final FieldConfigScheme configScheme = issueTypeSchemeManager.getConfigScheme(input);
				return configScheme != null ? configScheme.getId() : null;
			}
		});
		return ImmutableSet.copyOf(Iterables.filter(issueTypeSchemeIds, Predicates.notNull())).size() > 1;
	}

	protected boolean isMoreThanOneWorkflowSchemes(Iterable<Project> projects) {
		final Iterable<Long> workflowSchemesIds = Iterables.transform(projects, new Function<Project, Long>() {
			@Override
			public Long apply(Project input) {
				final AssignableWorkflowScheme workflowSchemeObj = workflowSchemeManager.getWorkflowSchemeObj(input);
				return workflowSchemeObj.getId() != null ? workflowSchemeObj.getId() : null;
			}
		});
		return ImmutableSet.copyOf(Iterables.filter(workflowSchemesIds, Predicates.notNull())).size() > 1;
	}

	protected Collection<Project> getExistingDestinationProjects() {
		Iterable<Project> existingProjects = Iterables.transform(getConfigBean().getSelectedProjects(), new Function<String, Project>() {
			@Override
			public Project apply(String input) {
				final String projectKey = getConfigBean().getProjectKey(input);
				if (projectKey == null) {
					return null;
				}
				return projectManager.getProjectObjByKey(projectKey);
			}
		});
		return ImmutableList.copyOf(Iterables.filter(existingProjects, Predicates.notNull()));
	}

	@Nullable
	protected GithubImportProcessBean getProcessBean() {
		final ImporterController controller = getController();

		if (controller == null) {
			throw new RuntimeException("Unknown Controller");
		}
		final ImportProcessBean processBean = controller.getImportProcessBeanFromSession();
		if (processBean == null || !(processBean instanceof GithubImportProcessBean)) {
			return null;
		}

		return (GithubImportProcessBean) controller.getImportProcessBeanFromSession();
	}

	@Override
	public ConfigBean getConfigBean() {
		return getProcessBean() != null ? (ConfigBean) getProcessBean().getConfigBean() : null;
	}

	@Override
	public String getFormTitle() {
		return getI18nHelper().getText("com.atlassian.jira.plugins.importer.github.step.projectmapping.form.title");
	}

	@Override
	public String getFormDescription() {
		return getI18nHelper().getText("com.atlassian.jira.plugins.importer.github.step.projectmapping.description");
	}
}
