package com.atlassian.jira.plugins.importer.github.fetch;

import com.atlassian.jira.component.ComponentAccessor;
import com.atlassian.jira.util.I18nHelper;
import org.apache.log4j.Logger;
import org.eclipse.egit.github.core.client.RequestException;

import javax.annotation.concurrent.ThreadSafe;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.FutureTask;
import java.util.concurrent.atomic.AtomicReference;

/**
 * A job that loads the data of all selected projects by calling {@link GithubDataService#loadProject} and records the progress.
 * Can be cancelled.
 */
@ThreadSafe
public class DataFetchJob {

    private final Logger log = Logger.getLogger(this.getClass());

    private final GithubDataService githubDataService;
    private final AtomicReference<FutureTask<Void>> futureTaskAtomicReference = new AtomicReference<FutureTask<Void>>();
    private final I18nHelper i18n;
	private FetchProgress fetchProgress;

	public DataFetchJob(GithubDataService githubDataService) {
        this.githubDataService = githubDataService;
        i18n = ComponentAccessor.getJiraAuthenticationContext().getI18nHelper();
    }

    /**
     * Perform the fetch for the given projects.
     * Does not throw an exception.
     */
    public void run(List<String> selectedProjects) {
		fetchProgress = new FetchProgress();
        FutureTask<Void> future = new FutureTask<Void>(new DataFetchCallable(selectedProjects, fetchProgress));
        if( futureTaskAtomicReference.compareAndSet(null, future) ) {
            try {
                log.info("Starting data fetch job");
                future.run();
            } finally {
                log.info("Data fetch job has stopped");
                futureTaskAtomicReference.set(null);
            }
        } else {
            throw new IllegalStateException("Fetch job is already running");
        }
    }

    /**
     * Cancels the fetch or does nothing if no fetch is running.
     */
    public void cancel() {
        FutureTask<Void> future = futureTaskAtomicReference.get();
        if( future != null ) {
            log.info("Cancelling data fetch job");
            future.cancel(true);
        }
		fetchProgress = new FetchProgress();
    }

    public boolean isRunning() {
        return futureTaskAtomicReference.get() != null;
    }

	public FetchProgress getFetchProgress() {
		return fetchProgress;
	}

	private class DataFetchCallable implements Callable<Void> {
        private final List<String> selectedProjects;
		private final FetchProgress fetchProgress;

        private DataFetchCallable(List<String> selectedProjects, FetchProgress fetchProgress) {
            this.selectedProjects = selectedProjects;
			this.fetchProgress = fetchProgress;
		}

        @Override
        public Void call() throws Exception {
            try {
                githubDataService.clearLoadedProjectData();
				fetchProgress.setTotalProjects(selectedProjects.size());
				int currentProjectNo = 1;
                // fetch all
                for(String projectName : selectedProjects ) {
                    githubDataService.loadProject(projectName, fetchProgress);
					fetchProgress.setCurrentProject(currentProjectNo++);
                }
				fetchProgress.setFinished(true);

            } catch (InterruptedException e) {
                githubDataService.clearLoadedProjectData();
                throw e;

            } catch (Exception e) {
                if( e instanceof RequestException && ((RequestException)e).getStatus() == 410 ) {
                    // issues are disabled in GitHub
                    fetchProgress.setError(i18n.getText("com.atlassian.jira.plugins.importer.github.fetchData.error.issuesDisabled"));
                    log.warn("Fetching data from GitHub failed", e);
                } else {
                    fetchProgress.setError(i18n.getText("com.atlassian.jira.plugins.importer.github.fetchData.error.otherExceptions"));
                    log.error("Fetching data from GitHub failed", e);
                }
            }
            return null;
        }
    }

}
