define("jira/editor/converter/handlers/list-item-handler", [
    "jira/editor/converter/handlers/node-handler",
    "jira/editor/converter/handlers/text-handler",
    "jira/editor/converter/handlers/empty-handler",
    "jira/editor/converter/handlers/list-type-matcher",
    "jira/editor/converter/util/strings"
], function (
    NodeHandler,
    TextHandler,
    EmptyHandler,
    ListTypeMatcher,
    Strings
) {
    "use strict";

    /**
     * @class ListItemHandler
     * @extends NodeHandler
     */
    return NodeHandler.extend({
        init: function ListItemHandler(node, depth, bulletDecorator) {
            this.node = node;
            if (typeof depth === 'undefined') {
                this.depth = 0;
            } else {
                this.depth = depth;
            }
            // one of '#', '*', '-'
            this.bulletDecorator = bulletDecorator;
        },

        before: function () {
            var asterisks = Strings.repeat(this.bulletDecorator, this.depth + 1);
            // statistically, most of the list items have an additional spacing before the bullet decorator.
            return TextHandler.createTextHandler(' ' + asterisks + ' ');
        },

        after: function () {
            return new EmptyHandler();
        },

        /**
         * Overrides processing each child node for lists.
         *
         * @param childNode child node that will be processed after this method
         * @param childIndex
         * @param childrenTotal
         * @returns empty or a text handler
         */
        createBeforeChild: function(childNode, childIndex, childrenTotal) {
            if (ListTypeMatcher.isListNode(childNode)) {
                // Before processing each list item 'li', adds a new line for each 'ul' item and other list items
                return TextHandler.newlineHandler();
            }
            return new EmptyHandler();
        },

        /**
         * Overrides building a child node.
         */
        createChildHandler: function(node, childIndex, childrenTotal) {
            var Handler = NodeHandler.getHandler(node);
            if (ListTypeMatcher.isListNode(node)) {
                // other handler processing
                return new Handler(node, this.depth + 1);
            } else {
                return new Handler(node);
            }
        }
    });
});