define("jira/editor/undomanager-adapter/text-adapter", [
    "jira/editor/constants",
    "jira/editor/marionette",
    "jira/util/logger",
    "jquery"
], function (
    EditorConstants,
    Marionette,
    logger,
    $
) {
    "use strict";

    var TextAdapter = Marionette.Object.extend({
        initialize: function initialize(options) {
            this.textEditor = $(options.textarea);
            var wikiEditor = this.textEditor.data('wikiEditor');
            if (!wikiEditor) {
                // it's happened, but hasn't been reproduced; see MNSTR-1275
                var message = "Wiki-editor should be already initialised. Could not find 'wikiEditor' instance in given" +
                    " textarea element"
                logger.error(message, this.textEditor);
                throw new Error(message);
            }
            this.textUndoableTextarea = wikiEditor.undoRedoEl;
            this.textUndoManager = this.textUndoableTextarea.undoManager;

            // It's needed for ignoring subsequent event coming from observed UndoManager
            //  when the action was triggered explicitly in "undo" or "redo" method.
            this.ongoingInternalOperation = false;

            this.attachEvents(this.textUndoManager);
        },
        undo: function undo() {
            try {
                this.ongoingInternalOperation = true;
                this.textUndoableTextarea.undo();
            } finally {
                this.ongoingInternalOperation = false;
            }
        },
        redo: function redo() {
            try {
                this.ongoingInternalOperation = true;
                this.textUndoableTextarea.redo();
            } finally {
                this.ongoingInternalOperation = false;
            }
        },
        hasUndo: function hasUndo() {
            return this.textUndoManager.canUndo();
        },
        hasRedo: function hasRedo() {
            return this.textUndoManager.canRedo();
        },
        clear: function clear() {
            this.textUndoManager.clear();
        },

        attachEvents: function attachEvents(textUndoManager) {
            var TextEventTypes = textUndoManager.EventTypes;
            textUndoManager.on([TextEventTypes.PUSH, TextEventTypes.UPDATE_CURRENT].join(' '), function(e) {
                if (!this.ongoingInternalOperation) {
                    this.triggerMethod("action", EditorConstants.EventTypes.ADD, EditorConstants.EventSources.UNKNOWN);
                }
            }.bind(this));
            textUndoManager.on(TextEventTypes.UNDO, function(e) {
                if (!this.ongoingInternalOperation) {
                    this.triggerMethod("action", EditorConstants.EventTypes.UNDO, EditorConstants.EventSources.SHORTCUT);
                }
            }.bind(this));
            textUndoManager.on(TextEventTypes.REDO, function(e) {
                if (!this.ongoingInternalOperation) {
                    this.triggerMethod("action", EditorConstants.EventTypes.REDO, EditorConstants.EventSources.SHORTCUT);
                }
            }.bind(this));
            textUndoManager.on(TextEventTypes.CLEAR, function(e) {
                if (!this.ongoingInternalOperation) {
                    this.triggerMethod("action", EditorConstants.EventTypes.CLEAR, EditorConstants.EventSources.UNKNOWN);
                }
            }.bind(this));
        }
    });

    return TextAdapter;
});
