define("jira/editor/ui/toggle-tabs", [
    "jira/editor/constants",
    "jira/editor/analytics",
    "jira/editor/toggle",
    "jira/editor/ui/editor-tabs-soy-amd",
    "jira/lib/class",
    "backbone",
    "underscore",
    "jquery"
], function(
    Constants,
    Analytics,
    editorToggle,
    editorTabs,
    Class,
    Backbone,
    _,
    $
) {
    var Modes = Constants.Modes;

    var ToggleTabs = Class.extend({
        /**
         * @param element
         */
        init: function(element) {
            this.$parent = $(element).parent('.wiki-edit-content');

            this.rendered = new $.Deferred();

            editorToggle.getMode().always(function(mode) {
                this.renderTabs(mode);
            }.bind(this));

            this.hideTooltip = _.debounce(this.hideTooltip, 750);
        },

        renderTabs: function(mode) {
            this.activeMode = mode;
            this.$tabs = $(editorTabs.tabs({
                tabs: [
                    { text: AJS.I18n.getText('jira.editor.toggle.visual'), url: '#', mode: Modes.VISUAL },
                    { text: AJS.I18n.getText('jira.editor.toggle.text'), url: '#', mode: Modes.TEXT }
                ],
                currentMode: mode
            }));

            this.$tabs.appendTo(this.$parent)
                .find('a').on('click', this.onTabSelect.bind(this));

            this.$tooltip = $(aui.labels.label({ text: '', extraClasses: 'editor-toggle-tooltip' }))
                .appendTo(this.$parent);

            this.rendered.resolve(this.$tabs);
        },

        onTabSelect: function(event) {
            event.preventDefault();
            var mode = $(event.target).parent().data("mode");
            if (mode && mode !== this.activeMode) {
                this._setModeInternal(mode);

                this.trigger("change", mode);
                editorToggle.setMode(mode);

                Analytics.sendEvent("editor.instance.switch.change");
                Analytics.sendEvent("bundled.editor.instance.switch.change");
            }
        },

        getMode: function() {
            var deferred = new $.Deferred();

            this.rendered.done(function($tabs) {
                var mode = $tabs.find(".aui-nav-selected").data("mode");

                if (mode) {
                    deferred.resolve(mode);
                } else {
                    deferred.reject();
                }
            }.bind(this));

            return deferred.promise();
        },

        setMode: function(mode) {
            this.rendered.done(function() {
                this._setModeInternal(mode);
            }.bind(this));
        },

        enable: function() {
            this.$tabs.removeClass('rte-tabs-disabled').find('a').removeAttr("tabindex");
        },

        disable: function() {
            this.$tabs.addClass('rte-tabs-disabled').find('a').attr("tabindex", -1);
        },

        showTooltip: function(mode) {
            var modeText = mode === Modes.VISUAL
                ? AJS.I18n.getText('jira.editor.toggle.mode.visual')
                : AJS.I18n.getText('jira.editor.toggle.mode.text');
            this.$tooltip
                .text(modeText)
                .addClass('visible');
        },

        hideTooltip: function() {
            this.$tooltip.removeClass('visible');
        },

        isVisible: function() {
            if (!this.$tabs) {
                return;
            }

            // element not present in DOM - rect will be null
            var rect = this.$tabs[0].getBoundingClientRect();
            // element not visible - elementFromPoint either null or is covered
            return rect && $(document.elementFromPoint(rect.left + rect.width / 2, rect.top + rect.height / 2)).parents().is(this.$tabs);
        },

        _setModeInternal: function(mode) {
            this.$tabs.find("li").removeClass("aui-nav-selected").filter(function(idx, el) {
                return $(el).data("mode") === mode;
            }).addClass("aui-nav-selected");

            this.activeMode = mode;

            this.showTooltip(mode);
            this.hideTooltip();
        }
    }).extend(Backbone.Events);

    return ToggleTabs;
});
