define("jira/editor/plugins/paste", [
    "jira/editor/tinymce",
    "jira/editor/converter/util/strings",
    "jira/editor/schema",
    "jira/editor/analytics",
    "jquery",
    "wrm/context-path",
    "dndattachment/ctrlv/utility",
    "dndattachment/ctrlv/base64decode",
    "jira/editor/wrm",
    "jira/flag",
    "jira/util/formatter"
], function (tinymce,
             Strings,
             EditorSchema,
             Analytics,
             $,
             wrmContextPath,
             pasteUtility,
             base64decode,
             wrmResources,
             JiraFlag,
             formatter) {
    "use strict";

    var isAttachmentsAllowed = wrmResources.isAttachmentsAllowed();

    tinymce.create('tinymce.plugins.jira.Paste', {});

    tinymce.PluginManager.add('jira.paste', tinymce.plugins.jira.Paste);
    tinymce.PluginManager.urls["jira.paste"] = true;

    function getContentTypeFromDataUrl(url) {
        if (url) {
            var header = url.split(',')[0];
            var headerSplit = header.split(':');
            if (headerSplit.length > 1) {
                return headerSplit[1].split(';')[0];
            }
        }
    }

    return {
        configure: function (instance, settings) {

            function getBlobFromTinymceCache(url) {
                var editor = instance.editor;
                if (editor && editor.editorUpload && editor.editorUpload.blobCache) {
                    return editor.editorUpload.blobCache.getByUri(url);
                }
            }

            var ImageType = {
                /**
                 * external url to image
                 */
                URL: {
                    upload: function () {

                    }
                },
                BLOB_URL: {
                    upload: function (data, fileName) {
                        Analytics.sendEvent("editor.instance.paste.addimage.nocache.http");
                        getAndUploadImage(data, fileName);
                    }
                },
                DATA_URL: {
                    upload: function (data, fileName) {
                        Analytics.sendEvent("editor.instance.paste.addimage.nocache.image");
                        var contentType = getContentTypeFromDataUrl(data) || 'image/png';
                        uploadImage(new Blob([base64decode.decodeBase64DataUri(data)], {type: contentType}), fileName);
                    }
                },
                TINYMCE_CACHED_BLOB: {
                    upload: function (data, fileName) {
                        var cachedBlob = getBlobFromTinymceCache(data, instance.editor);
                        if (cachedBlob) {
                            Analytics.sendEvent("editor.instance.paste.addimage.cache");
                            uploadImage(cachedBlob.blob(), fileName);
                        } else {
                            console.warn('no cached blob found in cache');
                        }
                    }
                }
            };

            function getImageType(src) {
                var cachedBlob = getBlobFromTinymceCache(src, instance.editor);
                if (cachedBlob) {
                    return ImageType.TINYMCE_CACHED_BLOB;
                } else if (src.match(/^((data|blob):http)|(blob:)/)) {
                    return ImageType.BLOB_URL;
                } else if (src.match(/^(data|blob):image/)) {
                    return ImageType.DATA_URL;
                }
                return ImageType.URL;
            }

            function uploadImage(data, fileName) {
                var file = pasteUtility.convertBlobToImage(data, fileName);
                if (!pasteUtility.dropFileToElement(file, instance.element.$textarea)) {
                    console.warn("Uploading attachment failed. Check 'allow attachment' setting.");
                    JiraFlag.showWarningMsg("", formatter.I18n.getText("jira.editor.tinymce.plugin.paste.no.handler"), {close: 'auto'});
                    instance.editor.dom.remove(instance.editor.dom.select('img[data-filename="' + fileName + '"]'));
                }
            }

            function getAndUploadImage(src, fileName) {
                var xhr = new XMLHttpRequest();
                xhr.open("GET", src, true);
                xhr.responseType = "blob";
                xhr.onload = function (e) {
                    uploadImage(this.response, fileName);
                };
                xhr.send();
            }

            settings.plugins.push('paste');
            settings.paste_data_images = true;
            settings.paste_preprocess = function (plugin, args) {
                var schema = instance.contextManager.getPasteSchemaForContext();
                args.content = EditorSchema.sanitizeHtml(args.content, instance.editor, schema);
            };
            settings.paste_postprocess = function (plugin, args) {
                Analytics.sendEvent("editor.instance.paste");
                var $image = $(args.node).find("img:first-child:last-child");
                if ($image.length === 1 && $image.is("img")) {
                    var src = $image[0].src;

                    if ($image.hasClass('emoticon') && Strings.startsWith(src, document.location.origin + wrmContextPath() + "/images/icons/emoticons/")) {
                        Analytics.sendEvent("editor.instance.paste.emoticon");
                        return;
                    }

                    Analytics.sendEvent("editor.instance.paste.image");

                    var filename = src;
                    if (Strings.isAttachmentPath(src) || Strings.isThumbnailPath(src)) {
                        filename = Strings.getFilenameFromHref(src);
                    }
                    var param = {
                        source: src,
                        filename: filename,
                        width: $image.attr('width'),
                        height: $image.attr('height'),
                        id: $image.attr('id'),
                        vspace: $image.attr('vspace')
                    };
                    var imageType = getImageType(src);
                    if (imageType === ImageType.DATA_URL || imageType === ImageType.BLOB_URL || imageType === ImageType.TINYMCE_CACHED_BLOB) {
                        if (!!isAttachmentsAllowed) {
                            filename = pasteUtility.generateFileName() + ".png";
                            imageType.upload(src, filename);
                            param.filename = filename;
                            var result = JIRA.Editor.Tags.Templates.image(param);
                            $image.replaceWith($(result));
                        } else {
                            JiraFlag.showWarningMsg("", formatter.I18n.getText("jira.editor.tinymce.plugin.paste.attachments.disabled"), {close: 'auto'});
                            $image.remove();
                        }
                    } else {
                        var result = JIRA.Editor.Tags.Templates.image(param);
                        $image.replaceWith($(result));
                    }
                } else {
                    Analytics.sendEvent("editor.instance.paste.other");
                }
            };

            return settings;
        }
    };
});