AJS.test.require([
    'com.atlassian.jira.plugins.jira-editor-plugin:context-detector'
], function () {

    var $ = require('jquery');

    /**
     * Call it with the module context.
     */
    function setup() {
        this.sandbox = sinon.sandbox.create();

        this.ContextDetector = require('jira/editor/context-detector');
    }

    module('ContextDetector "pre"', {
        setup: function() {
            setup.call(this);

            this.containers = [
                $('<pre></pre>'),
                $('<pre class="noformat panel"></pre>'),
                $('<pre class="code panel"></pre>'),
                $('<div class="code panel"></div>'),
                $('<div class="preformatted panel"></div>'),
                $('<pre class="preformatted panel"></pre>'),
                $('<pre class="panelHeader"></pre>'),
                $('<div></div><pre></pre>')
            ];
            this.content = $('<div class="fancy-style">text</div>');
            this.panelTitle = $('<panel-title>title</panel-title>');
        },

        teardown: function () {
            this.sandbox.restore();
        }
    });

    test('Should detect "pre" context when container is selected', function () {
        this.containers.forEach(function(cont) {
            ok(this.ContextDetector.detectPre(cont), '"pre" context should be detected for node ' + decorateNode(cont));
        }, this);
    });

    test('Should detect "pre" context when paneltitle is selected', function () {
        this.containers.forEach(function(cont) {
            cont.append(this.panelTitle);
            cont.append(this.content);
            ok(this.ContextDetector.detectPre(this.panelTitle), '"pre" context should be detected for node ' + decorateNode(this.panelTitle));
        }, this);
    });

    test('Should detect "pre" context when content is selected', function () {
        this.containers.forEach(function(cont) {
            cont.append(this.content);
            ok(this.ContextDetector.detectPre(this.content), '"pre" context should be detected for node ' + decorateNode(this.content));
        }, this);
    });

    test('Should not detect "pre" context #1', function () {
        var nodes = $('<div class="code panel">pre context</div><table><tbody><tr><td>table context</td></tr></tbody></table>');
        var targetNode = nodes.find('td');
        ok(!this.ContextDetector.detectPre(targetNode), '"pre" context should not be detected for tree ' + decorateTree(nodes));
    });

    test('Should not detect "pre" context #2', function () {
        var nodes = $('<div class="panel"><span>no "pre" context</span></div>');
        var targetNode = nodes.find('span');
        ok(!this.ContextDetector.detectPre(targetNode), '"pre" context should not be detected for tree ' + decorateTree(nodes));
    });


    module('ContextDetector "table"', {
        setup: function() {
            setup.call(this);

            this.containers = [
                $('<table></table>'),
                $('<div></div><table></table>'),
                $('<table><tbody><tr><td></td></tr></tbody></table>'),
                $('<table><thead><tr><th>title</th></tr></thead><tbody><tr><td>table context</td></tr></tbody></table>')
            ];
        },

        teardown: function () {
            this.sandbox.restore();
        }
    });

    test('Should detect "table" context when container is selected', function () {
        this.containers.forEach(function(cont) {
            ok(this.ContextDetector.detectTable(cont), '"table" context should be detected for node ' + decorateNode(cont));
        }, this);
    });

    test('Should detect "table" context when cell is selected', function () {
        this.containers.forEach(function(cont) {
            var targetNode = cont.find('td');
            if (targetNode.length) {
                ok(this.ContextDetector.detectTable(targetNode), '"table" context should be detected for node ' + decorateNode(targetNode));
            }

            var targetNode = cont.find('th');
            if (targetNode.length) {
                ok(this.ContextDetector.detectTable(targetNode), '"table" context should be detected for node ' + decorateNode(targetNode));
            }
        }, this);
    });

    test('Should detect "table" context from complex cell content', function () {
        var nodes = $('<table><tbody><tr><td><div><strong><em>content</em></strong><pre><span>pre</span></pre></div></td></tr></tbody></table>');
        var targetNode = nodes.find('em');
        if (targetNode.length) {
            ok(this.ContextDetector.detectTable(targetNode), '"table" context should be detected for node ' + decorateNode(targetNode));
        }

        var targetNode = nodes.find('pre');
        if (targetNode.length) {
            ok(this.ContextDetector.detectTable(targetNode), '"table" context should be detected for node ' + decorateNode(targetNode));
        }
    });

    test('Should not detect "table" context #1', function () {
        var nodes = $('<div><div class="code panel">pre context</div></div><table><tbody><tr><td>table context</td></tr></tbody></table>');
        var targetNode = nodes.find('div.code.panel');
        ok(!this.ContextDetector.detectTable(targetNode), '"table" context should not be detected for tree ' + decorateTree(nodes));
    });

    test('Should not detect "table" context #2', function () {
        var nodes = $('<div style="display:table"><div style="display:table-row"><div style="display:table-cell"><span class="css-table">css table</span></div></div></div>');
        var targetNode = nodes.find('span.css-table');
        ok(!this.ContextDetector.detectTable(targetNode), '"table" context should not be detected for tree ' + decorateTree(nodes));
    });


    module('ContextDetector "a"', {
        setup: function() {
            setup.call(this);

            this.containers = [
                $('<a></a>'),
                $('<a name="anchor">link</a>'),
                $('<a href="#">link</a>'),
                $('<div></div><a></a>')
            ];

            this.content = $('<strong><em>link</em></strong>');
        },

        teardown: function () {
            this.sandbox.restore();
        }
    });

    test('Should detect "a" context when container is selected', function () {
        this.containers.forEach(function(cont) {
            ok(this.ContextDetector.detectA(cont), '"a" context should be detected for node ' + decorateNode(cont));
        }, this);
    });

    test('Should detect "a" context when content is selected', function () {
        this.containers.forEach(function(cont) {
            cont.append(this.content);
            ok(this.ContextDetector.detectA(this.content), '"a" context should be detected for node ' + decorateNode(this.content));
        }, this);
    });

    test('Should not detect "a" context', function () {
        var nodes = $('<div><div class="a">pre context</div></div><a></a>');
        var targetNode = nodes.find('div.a');
        ok(!this.ContextDetector.detectTable(targetNode), '"a" context should not be detected for tree ' + decorateTree(nodes));
    });

    test('Should not detect "a" context for node whose child has that context', function () {
        var nodes = $('<div><div class="a"><a>link</a></div></div>');
        var targetNode = nodes.find('div.a');
        ok(!this.ContextDetector.detectTable(targetNode), '"a" context should not be detected for tree ' + decorateTree(nodes));
    });


    module('ContextDetector all contexts combined', {
        setup: function() {
            setup.call(this);

            this.trees = [
                $('<pre><table><tr><td><a><div></div></a></td></tr></table></pre>'),
                $('<pre><a><table><tr><td><div></div></td></tr></table></a></pre>'),
                $('<table><tr><td><pre><a><div></div></a></pre></td></tr></table>'),
                $('<table><tr><td><a><pre><div></div></pre></a></td></tr></table>'),
                $('<a><pre><table><tr><td><div></div></td></tr></table></pre></a>'),
                $('<a><table><tr><td><pre><div></div></pre></td></tr></table></a>'),
            ];
        },

        teardown: function () {
            this.sandbox.restore();
        }
    });

    test('Should detect "pre" context when container is selected', function () {
        this.trees.forEach(function(tree) {
            var targetNode = tree.find('div');
            ok(this.ContextDetector.detectPre(targetNode), '"pre" context should be detected for tree ' + decorateTree(tree));
            ok(this.ContextDetector.detectTable(targetNode), '"table" context should be detected for tree ' + decorateTree(tree));
            ok(this.ContextDetector.detectA(targetNode), '"a" context should be detected for tree ' + decorateTree(tree));
        }, this);
    });


    module('ContextDetector "pre" selection', {
        setup: function() {
            setup.call(this);

            this.goodTrees = [
                $('<div><pre></pre></div>'),
                $('<div><div><pre class="noformat panel"></pre></div><div><strong><a>link</a></strong></div>'),
                $('<div><pre class="code panel"></pre></div>'),
                $('<div><div class="code panel"></div></div>'),
                $('<div><div class="preformatted panel"></div></div>'),
                $('<div><pre class="preformatted panel"></pre></div>'),
                $('<div><div class="panelHeader"></div></div>'),
                $('<div></div><div><pre></pre></div>')
            ];

            this.badTrees = [
                $('<pre></pre>'),
                $('<pre class="noformat panel"></pre>'),
                $('<pre class="code panel"></pre>'),
                $('<div class="code panel"></div>'),
                $('<div class="preformatted panel"></div>'),
                $('<pre class="preformatted panel"></pre>'),
                $('<div></div><pre></pre>')
            ];

            this.maliciousNode = '%MCECOPIED%';
        },

        teardown: function () {
            this.sandbox.restore();
        }
    });

    test('Should detect "pre" within selection', function () {
        this.goodTrees.forEach(function(tree) {
            ok(this.ContextDetector.detectPreWithinSelection(tree), '"pre" within selection should be detected for tree ' + decorateTree(tree));
        }, this);
    });

    test('Should not detect "pre" within selection', function () {
        this.badTrees.forEach(function(tree) {
            ok(!this.ContextDetector.detectPreWithinSelection(tree), '"pre" within selection should not be detected for tree ' + decorateTree(tree));
        }, this);

        ok(!this.ContextDetector.detectPreWithinSelection(this.maliciousNode), '"pre" within selection should not be detected when node cannot be parsed');
    });

    test("Should not fail on inline text", function() {
        ok(!this.ContextDetector.detectPreWithinSelection('workflow documentation for end users ("&lt;a target=\"_blank\" href=\"https://confluence.atlassian.com/display/JIRA/Configuring+Workflow'));
    });

    function decorateNode(node) {
        return $(node).get(0).cloneNode(false).outerHTML;
    }

    function decorateTree(tree) {
        return $(tree).get(0).outerHTML;
    }
});