AJS.test.require("com.atlassian.jira.plugins.jira-development-integration-plugin:devstatus-panel-resources");

(function() {

    module("JIRA.DevStatus.DevStatusModule", {
        setup: function() {
            this.sandbox = sinon.sandbox.create();

            this.fixture = jQuery("#qunit-fixture");
            this.devStatusPanel = jQuery('<div id="devstatus-panel"></div>').appendTo(this.fixture);
            this.devStatusContainer = jQuery('<div id="devstatus-container"></div>').appendTo(this.devStatusPanel);
            this.statusPanels = jQuery('<div class="status-panels"></div>').appendTo(this.devStatusContainer);
            this.statusPanelContainer = jQuery('<div><div class="status-panel" data-module="MockModule"></div></div>').appendTo(this.statusPanels);
            this.messagePanelContainer = jQuery('<div class="message-panel"></div>').appendTo(this.devStatusContainer);

            this.backupEverything();
            this.stubEverything();
        },
        teardown : function () {
            this.sandbox.restore();
            AJS.$.ajax = this.ajax;
            AJS.$("body").removeClass();
        },
        backupEverything: function() {
            this.ajax = AJS.$.ajax;
        },
        stubEverything: function() {
            AJS.$.ajax = this.sandbox.stub();
        },

        /**
         * @param {object} [opts]
         * @param {object} [opts.labs] the labs configuration
         *
         * @returns {JIRA.DevStatus.DevStatusModule}
         */
        createModule: function(opts) {
            if (opts && opts.labs) {
                this.devStatusContainer.attr('data-labs-json', JSON.stringify(opts.labs));
            }

            var module = createLightWeightModule();
            JIRA.DevStatus.MockModule = function() {
                return {
                    render: sinon.spy(),
                    startLoading: sinon.spy()
                };
            };

            // we shouldn't be reaching into the innards of DevStatusModule like this but
            // I am leaving this for backward compatibility with existing tests.
            module.statusPanelContainer = this.statusPanelContainer;
            module.devStatusContainer = { data: sinon.spy(), append: sinon.spy() }; // required by SummaryErrorModule
            module._createDevStatusData({});
            module._createAnalyticsModel({});
            module._createSummaryViews({});

            module._errorModule.render = sinon.spy();
            module._postRender = sinon.spy();
            return module;
        },

        createModuleForNoData: function() {
            var module = createLightWeightModule();
            module.analyticsModel = {getIssue: function(){return {}}, getSummary: sinon.stub()};
            module._isAnySummaryViewVisibleOnPanel = sinon.stub();
            module._isAnyErrorViewVisible = sinon.stub();
            module.devStatusPanelVisibilityToggler = sinon.stub();
            module.parentContainer = { is: sinon.stub() };

            return module;
        }
    });

    var selectors = {
        devStatusContainer: "#qunit-fixture #devstatus-container",
        parentContainer: "[id$='devstatus-panel']",
        link: "a.devstatus-cta-link",
        statusPanel: ".status-panels",
        viewIssueHeader: ".mod-header",
        agileHeader: ".ghx-header"
    };
    var darkFeatures = {
        labs: "jira.plugin.devstatus.labs",
        phaseTwo: "jira.plugin.devstatus.phasetwo"
    };

    function createLightWeightModule() {
        var module = new JIRA.DevStatus.DevStatusModule({
            parentContainerSelector: selectors.parentContainer,
            linkSelector: selectors.link,
            panelSelector: selectors.devStatusContainer,
            statusPanelSelector: selectors.statusPanel,
            enableLabs: true,
            enablePhaseTwo: true,
            darkFeatures: darkFeatures
        });
        module.parentContainer = jQuery();
        module.labsOptIn = new JIRA.DevStatus.Labs.LabsOptInModel({ allowed: true });
        return module;
    }

    function mockAjax() {
        var deferred = AJS.$.Deferred();
        AJS.$.ajax.returns(deferred);
        return deferred;
    }

    /**
     * Returns a new div containing the JSON blob.
     *
     * @param {object} devSummary a dev summary to be inserted into the div
     * @returns {jQuery}
     */
    function createJsonBlob(devSummary) {
        return jQuery('<div class="dev-summary json-blob"></div>').attr('data-json', JSON.stringify(devSummary))
    }

    test("ajax successful", function() {
        var module = this.createModule(),
            ajaxDeferred = mockAjax();

        module._setupDataEventHandlers();
        module._startDevStatus();
        var aggregatedData = [ {instance: {name:"stash1"}, data: {}}];
        ajaxDeferred.resolve(aggregatedData);
        ok(module._summaryModules[0].render.withArgs(aggregatedData).calledOnce, "first module is rendered");
        ok(module._errorModule.render.withArgs(aggregatedData).calledOnce, "error module is rendered");
        ok(module._postRender.withArgs().calledOnce, "no data module is rendered");
    });

    test("ajax failed", function() {
        var module = this.createModule(),
            ajaxDeferred = mockAjax();

        module._setupDataEventHandlers();
        module._startDevStatus();
        ajaxDeferred.reject();
        ok(module._summaryModules[0].render.withArgs().calledOnce, "first module is rendered with no data");
        ok(module._errorModule.render.withArgs().calledOnce, "error module is rendered with no data");
        ok(module._postRender.withArgs().calledOnce, "no data module is rendered");
    });

    test("no AJAX call if fresh dev summary JSON is found on the page", function() {
        this.devStatusContainer.append(createJsonBlob({
            isStale: false,
            cachedValue: {instance: {name:"stash1"}, data: {}}
        }));

        var devStatusModule = this.createModule();
        var ajaxDeferred = mockAjax();

        devStatusModule.initializePanel({
            container: this.devStatusContainer,
            showTooltip: false,
            headerSelector: ''
        });

        devStatusModule._startDevStatus();
        equal(AJS.$.ajax.callCount, 0, "DevStatusModule should use JSON blob that is on the page");
    });

    test("AJAX call is still performed when stale dev summary JSON is found on the page", function() {
        this.devStatusContainer.append(createJsonBlob({
            isStale: true,
            cachedValue: {instance: {name: "stash1"}, data: {}}
        }));

        var devStatusModule = this.createModule();
        var ajaxDeferred = mockAjax();

        devStatusModule.initializePanel({
            container: this.devStatusContainer,
            showTooltip: false,
            headerSelector: ''
        });

        equal(AJS.$.ajax.callCount, 1, "DevStatusModule should use JSON blob that is on the page");
    });

    test("stale dev summary data is not cleared when update summary data AJAX fails", function() {
        var staleData = {instance: {name: "stash1"}, data: {}};
        this.devStatusContainer.append(createJsonBlob({
            isStale: true,
            cachedValue: staleData
        }));

        // reject the AJAX
        mockAjax().reject();

        var devStatusModule = this.createModule().initializePanel({
            container: this.devStatusContainer
        });

        ok(this.fixture.find('.connection-error').length == 1, "comms error message should be shown");
        ok(devStatusModule._summaryModules[0].render.withArgs(staleData).calledTwice, "stale summary data should be shown");
    });

    test("AJAX call is not performed when phase two is disabled", function() {
        this.devStatusContainer.append(createJsonBlob({
            isStale: true,
            cachedValue: {instance: {name:"stash1"}, data: {}}
        }));

        var devStatusModule = this.createModule();
        var ajaxDeferred = mockAjax();

        devStatusModule.initializePanel({
            container: this.devStatusContainer,
            showTooltip: false,
            headerSelector: '',
            phaseTwoDisabled: true
        });

        equal(AJS.$.ajax.callCount, 0, "phase two ajax call is not made");
    });

    test("is any view visible", function() {
        var module = createLightWeightModule();

        module.labsOptIn = new JIRA.DevStatus.Labs.LabsOptInModel({ allowed: false });
        module._summaryModules = [ {isViewVisible: function() { return true;}}];
        ok(module._isAnySummaryViewVisibleOnPanel(), "visible even if labs not allowed");

        module.labsOptIn = new JIRA.DevStatus.Labs.LabsOptInModel({ allowed: true, optedIn: true});
        module._summaryModules = [ {isViewVisible: function() { return false;}}];
        ok(!module._isAnySummaryViewVisibleOnPanel(), "not visible even if labs opted in");

        module._summaryModules = [];
        ok(!module._isAnySummaryViewVisibleOnPanel(), "not visible due to no modules");

        module._summaryModules = [ {isViewVisible: function() { return false;}}];
        ok(!module._isAnySummaryViewVisibleOnPanel(), "not visible due to no modules visible");

        module._summaryModules = [ {isViewVisible: function() { return false;}}, {isViewVisible: function() { return true;}}];
        ok(module._isAnySummaryViewVisibleOnPanel(), "visible due to some modules visible");
    });

    test("is any error view visible", function() {
        var module = createLightWeightModule();

        module._errorModule = {isViewVisible: function() { return true;}};
        ok(module._isAnyErrorViewVisible(), "visible if error is visible");

        module._errorModule = {isViewVisible: function() { return false;}};
        ok(!module._isAnyErrorViewVisible(), "not visible if error is not visible");
    });

    test("visibility toggler called with visible when summary view is displayed and container is NOT visible", function() {
        var module = this.createModuleForNoData();

        module._isAnySummaryViewVisibleOnPanel.returns(true);
        module.parentContainer.is.returns(false);
        module._postRender(true);
        ok(module.devStatusPanelVisibilityToggler.calledWith(module.parentContainer, true, true), "visibility toggler called with visible when summary view is displayed and container is NOT visible");
    });

    test("visibility toggler NOT called with visible when summary view is displayed and container is visible", function() {
        var module = this.createModuleForNoData();
        module._isAnySummaryViewVisibleOnPanel.returns(true);
        module.parentContainer.is.returns(true);
        module._postRender(true);
        ok(module.devStatusPanelVisibilityToggler.callCount === 0, "visibility toggler NOT called with visible when summary view is displayed and container is visible");
    });

    test("visibility toggler called with visible when summary view is not displayed but with create branch and container is NOT visible", function() {
        var module = this.createModuleForNoData();
        module._isAnySummaryViewVisibleOnPanel.returns(false);
        module.createBranchContainer = {size: function() {return 1;}};
        module.parentContainer.is.returns(false);
        module._postRender(false);
        ok(module.devStatusPanelVisibilityToggler.calledWith(module.parentContainer, true, false), "visibility toggler called with visible when summary view is not displayed but with create branch and container is NOT visible");
    });

    test("set the empty-status flag when nothing is rendered for the status panel", function() {
        var panel = AJS.$("<div></div>");
        var module = this.createModuleForNoData();
        module.statusPanelContainer = panel;
        module.createBranchContainer = {size: function() {return 1;}};

        module._isAnySummaryViewVisibleOnPanel.returns(true);
        module._postRender(true);
        equal(panel.hasClass("empty-status"), false, "panel doesn't have empty-status when something is being rendered");

        module._isAnySummaryViewVisibleOnPanel.returns(false);
        module._postRender(true);
        equal(panel.hasClass("empty-status"), true, "panel HAVE empty-status when nothing is being rendered");
    });

    test("visibility toggler NOT called with visible when summary view is not displayed but with create branch and container is visible", function() {
        var module = this.createModuleForNoData();
        module._isAnySummaryViewVisibleOnPanel.returns(false);
        module.createBranchContainer = {size: function() {return 1;}};
        module.parentContainer.is.returns(true);
        module._postRender(false);
        ok(module.devStatusPanelVisibilityToggler.callCount === 0, "visibility toggler NOT called with visible when summary view is not displayed but with create branch and container is visible");
    });

    test("visibility toggler called with not visible when summary view is not displayed and no create branch and container is visible", function() {
        var module = this.createModuleForNoData();
        module._isAnySummaryViewVisibleOnPanel.returns(false);
        module.createBranchContainer = {size: function() {return 0;}};
        module.parentContainer.is.returns(true);
        module._postRender(false);
        ok(module.devStatusPanelVisibilityToggler.calledWith(module.parentContainer, false, false), "visibility toggler called with not visible when summary view is not displayed and no create branch and container is visible");
    });

    test("visibility toggler NOT called with not visible when summary view is not displayed and no create branch and container is NOT visible", function() {
        var module = this.createModuleForNoData();
        module._isAnySummaryViewVisibleOnPanel.returns(false);
        module.createBranchContainer = {size: function() {return 0;}};
        module.parentContainer.is.returns(false);
        module._postRender(false);
        ok(module.devStatusPanelVisibilityToggler.callCount === 0, "visibility toggler NOT called with not visible when summary view is not displayed and no create branch and container is NOT visible");
    });

    test("calling initializePanel unregisters previously active listeners", function() {
        // make sure Labs is enabled otherwise nothing happens
        var devStatusModule = this.createModule({
            labs: {
                'allowed': true,
                'optedIn': true,
                'optedInByAdmin': false,
                'dismissed': false
            }
        });

        // initialise the panel for the  1st issue
        var ajax1 = mockAjax();
        devStatusModule.initializePanel({ container: this.devStatusContainer, showTooltip: false, headerSelector: '' });

        // initialise the panel a 2nd time--listeners should be reset when this happens
        var ajax2 = mockAjax();
        devStatusModule.initializePanel({ container: this.devStatusContainer, showTooltip: false, headerSelector: '' });

        // now resolve the 1st request with data. we are no longer on that issue so this should be a no-op
        ajax1.resolve([ {instance: {name:"stash1"}, data: {}}]);
        ok(this.fixture.find('.connection-error').length == 0, "comms error message should not be shown if AJAX for non-visible panel succeeds");

        ajax2.reject();
        ok(this.fixture.find('.connection-error').length == 1, "comms error message should be shown if AJAX fails");
    });

})();
