(function ($, JIRA) {
    AJS.namespace("JIRA.DevStatus.Analytics");

    var editPermissionLinkSelector = ".edit-permission-link";

    function isAnalyticsEnabled() {
        return AJS.EventQueue && !AJS.DarkFeatures.isEnabled("jira.plugin.devstatus.analytics.disabled");
    }

    JIRA.DevStatus.Analytics = {
        /**
         * Convenience to create and return an object that represents a devstatus action.
         *
         * @param name  a unique name for the event, could be namespaced with ".", e.g., "createbranch.click"
         *              will be prepended with "devstatus."
         * @param opts  an object containing any particular properties of relevance for this navigation type.
         */
        fireEvent : function (name, opts) {
            if (isAnalyticsEnabled()) {
                var props = (typeof opts !== "object") ? {} : opts;
                AJS.EventQueue.push({ name: "devstatus." + name, properties: props });
            }
        },
        fireAgileDevStatusLinkClicked : function() {
            return JIRA.DevStatus.Analytics.fireEvent("panel.agile.icon.clicked");
        }
    };

    JIRA.DevStatus.LabsAnalytics = {
        fireLabsEvent: function(toggle) {
            var msg = toggle ? "on" : "off";
            return JIRA.DevStatus.Analytics.fireEvent("labs.toggle." + msg);
        }
    };

    /**
     * Summary shown:
     *   * Current user is assigned: devstatus.summary.shown.assignee
     *   * Current user it NOT assigned: devstatus.summary.shown.nonassignee
     */
    JIRA.DevStatus.SummaryAnalytics = {
        fireSummaryShownEvent: function(isAssignee, issueType, issueStatus, summarisedData) {
            var assigneeText = isAssignee ? "assignee" : "nonassignee";
            var opts = _.extend({
                "issueType": issueType,
                "issueStatus": issueStatus
            }, summarisedData);
            return JIRA.DevStatus.Analytics.fireEvent("summary.shown." + assigneeText, opts);
        },
        fireSummaryClickedEvent: function(infoType, summarisedData) {
            var isNewContext = AJS.Meta.get("fusion-analytics-new-context-link");
            var event = ".clicked";
            if (isNewContext) {
                AJS.Meta.set("fusion-analytics-new-context-link", undefined);
                event += ".newcontext"
            }
            return JIRA.DevStatus.Analytics.fireEvent(infoType + ".summary" + event, summarisedData);
        },
        fireDetailTabClicked: function(applicationType, dataType) {
            return JIRA.DevStatus.Analytics.fireEvent(dataType + ".detail." + applicationType + ".tab.clicked");
        }
    };

    JIRA.DevStatus.BuildsAnalytics = {
        fireSummaryClickedEvent: function(summarisedData) {
            return JIRA.DevStatus.SummaryAnalytics.fireSummaryClickedEvent("builds", summarisedData);
        },
        fireDetailProjectClicked: function() {
            return JIRA.DevStatus.Analytics.fireEvent("builds.detail.project.clicked");
        },
        fireDetailPlanClicked: function() {
            return JIRA.DevStatus.Analytics.fireEvent("builds.detail.plan.clicked");
        },
        fireDetailBuildClicked: function() {
            return JIRA.DevStatus.Analytics.fireEvent("builds.detail.build.clicked");
        }
    };

    JIRA.DevStatus.DeploymentsAnalytics = {
        fireSummaryClickedEvent: function(summarisedData) {
            return JIRA.DevStatus.SummaryAnalytics.fireSummaryClickedEvent("deployments", summarisedData);
        },
        fireDetailProjectClicked: function() {
            return JIRA.DevStatus.Analytics.fireEvent("deployments.detail.project.clicked");
        },
        fireDetailEnvironmentClicked: function() {
            return JIRA.DevStatus.Analytics.fireEvent("deployments.detail.environment.clicked");
        },
        fireDetailReleaseClicked: function() {
            return JIRA.DevStatus.Analytics.fireEvent("deployments.detail.release.clicked");
        }
    };

    JIRA.DevStatus.CommitsAnalytics = {
        fireSummaryClickedEvent: function(summarisedData) {
            return JIRA.DevStatus.SummaryAnalytics.fireSummaryClickedEvent("commits", summarisedData);
        },
        fireDetailRepoClicked: function(type) {
            return JIRA.DevStatus.Analytics.fireEvent("commits.detail." + type + ".repo.clicked");
        },
        fireDetailCommitClicked: function(type) {
            return JIRA.DevStatus.Analytics.fireEvent("commits.detail." + type + ".commit.clicked");
        },
        fireDetailTabClicked: function(type) {
            return JIRA.DevStatus.SummaryAnalytics.fireDetailTabClicked(type, "commits");
        },
        fireDetailFileExpandedClicked: function(type) {
            return JIRA.DevStatus.Analytics.fireEvent("commits.detail." + type + ".file.expanded.clicked");
        },
        fireDetailFilesExpandedClicked: function(type) {
            return JIRA.DevStatus.Analytics.fireEvent("commits.detail." + type + ".files.expanded.clicked");
        },
        fireDetailFileClicked: function(type) {
            return JIRA.DevStatus.Analytics.fireEvent("commits.detail." + type + ".file.clicked");
        },
        fireDetailReviewsShown: function(type) {
            return JIRA.DevStatus.Analytics.fireEvent('commits.detail.' + type + ".reviews.shown");
        },
        fireDetailReviewClicked: function (type) {
            return JIRA.DevStatus.Analytics.fireEvent('commits.detail.' + type + ".review.clicked");
        },
        fireDetailBranchesShown: function(type) {
            return JIRA.DevStatus.Analytics.fireEvent('commits.detail.' + type + ".branches.shown");
        },
        fireDetailCreateReviewClicked: function(type, allCommits) {
            return JIRA.DevStatus.Analytics.fireEvent('commits.detail.' + type + '.review.create' + (allCommits ? '.all' : ''));
        }
    };

    JIRA.DevStatus.ReviewsAnalytics = {
        fireSummaryClickedEvent: function(summarisedData) {
            return JIRA.DevStatus.SummaryAnalytics.fireSummaryClickedEvent("reviews", summarisedData);
        },
        fireDetailReviewClicked: function() {
            return JIRA.DevStatus.Analytics.fireEvent("reviews.detail.review.clicked");
        }
    };

    /**
     * Branches:
     *
     * Summary clicked: devstatus.branches.summary.clicked
     * Detail:
     *   Repo: devstatus.branches.detail.bitbucket.repo.clicked
     *   Branch: devstatus.branches.detail.bitbucket.branch.clicked
     *   PR lozenge: devstatus.branches.detail.bitbucket.pullrequest.lozenge.clicked
     *   Create pull request:
     *     Current user is assigned: devstatus.branches.detail.bitbucket.pullrequest.create.assignee.clicked
     *     Current user is not assigned:
     *        Is assignable: devstatus.branches.detail.bitbucket.pullrequest.create.nonassignee.assignable.clicked
     *        Not assignable: devstatus.branches.detail.bitbucket.pullrequest.create.nonassignee.nonassignable.clicked
     *   Product tab: devstatus.branches.detail.bitbucket.tab.clicked
     *
     * Pull requests:
     * Summary clicked: devstatus.pullrequests.summary.clicked
     * Detail:
     *   Pull request link: devstatus.pullrequests.detail.bitbucket.pullrequest.clicked
     *   Product tab: devstatus.pullrequests.detail.bitbucket.tab.clicked
     *
     */
    JIRA.DevStatus.BranchesAnalytics = {
        fireSummaryClickedEvent: function(summarisedData) {
            return JIRA.DevStatus.SummaryAnalytics.fireSummaryClickedEvent("branches", summarisedData);
        },
        _fireDetailClickedEvent: function(applicationType, eventType) {
            return JIRA.DevStatus.Analytics.fireEvent("branches.detail." + applicationType + "." + eventType + ".clicked");
        },
        fireDetailRepoClicked: function(type) {
            this._fireDetailClickedEvent(type, "repo");
        },
        fireDetailBranchClicked: function(type) {
            this._fireDetailClickedEvent(type, "branch");
        },
        fireDetailPullRequestLozengeClick: function(type) {
            this._fireDetailClickedEvent(type, "pullrequest.lozenge");
        },
        fireDetailTabClicked: function(type) {
            return JIRA.DevStatus.SummaryAnalytics.fireDetailTabClicked(type, "branches");
        },
        fireDetailCreatePullRequestClicked: function(type, isAssignee, isAssignable) {
            var eventName = "pullrequest.create";
            if (isAssignee) {
                eventName += ".assignee";
            } else {
                eventName += ".nonassignee";
                if (isAssignable) {
                    eventName += ".assignable"
                } else {
                    eventName += ".nonassignable"
                }
            }
            this._fireDetailClickedEvent(type, eventName);
        }
    };

    JIRA.DevStatus.PullRequestsAnalytics = {
        fireSummaryClickedEvent: function(summarisedData) {
            return JIRA.DevStatus.SummaryAnalytics.fireSummaryClickedEvent("pullrequests", summarisedData);
        },
        fireDetailPullRequestClicked: function(type) {
            return JIRA.DevStatus.Analytics.fireEvent("pullrequests.detail." + type + ".pullrequest.clicked");
        },
        fireDetailTabClicked: function(type) {
            return JIRA.DevStatus.SummaryAnalytics.fireDetailTabClicked(type, "pullrequests");
        }
    };

    JIRA.DevStatus.LearnMoreAnalytics = {
        fireLearnMoreEvent: function (name, opts) {
            return JIRA.DevStatus.Analytics.fireEvent("learnmore." + name, opts);
        }
    };

    JIRA.DevStatus.CtaDialogAnalytics = {
        issueDetails : {}, // updated each time the create branch button is clicked

        fireCreateBranchEvent : function (name, opts) {
            opts = (typeof opts !== "object") ? {} : opts;
            // add issue status and issue type
            var props = $.extend({
                issueStatus : this.issueDetails.status,
                issueType : this.issueDetails.type
            }, opts);
            return JIRA.DevStatus.Analytics.fireEvent("createbranch." + name + "." + (this.issueDetails.isAssignee ? "assignee" : "nonassignee"), props);
        },

        _populateIssueDetails : function (link) {
            this.issueDetails = {
                status : "" + link.data('issueStatus'), // to undo potential type conversion done by .data()
                type : "" + link.data('issueType'),
                isAssignee : link.data('isAssignee')
            };
        },

        /**
         * Pull out the handler as an independent function so that the tests could unbind it in tear down.
         * @param e the event. event.data should contains instance, panelSelector and linkSelector
         * @param context
         * @private
         */
        _initHandler : function (e, context, panelSelector, linkSelector) {
            if (panelSelector && context.not(document).is(panelSelector)) {
                var instance = this;
                // Capture clicks on create branch button on view issue page
                context.find(linkSelector).on("simpleClick", function () {
                    var el = $(this);
                    instance._fireCreateBranchEvent(el);
                });
            }
        },

        _fireCreateBranchEvent : function (el) {
            this._populateIssueDetails(el);
            /**
             * Whether the create branch link is being clicked in a new context (ie in a new tab)
             */
            var eventName = "click";
            var isNewContext = AJS.Meta.get("fusion-analytics-new-context-link");
            if (isNewContext) {
                AJS.Meta.set("fusion-analytics-new-context-link", undefined);
                eventName += ".newcontext";
            }
            this.fireCreateBranchEvent(eventName);
        },

        initialize : function (panelSelector, linkSelector, dialogId) {
            if (isAnalyticsEnabled()) {
                var instance = this;

                // for issue nav view
                // bind the create branch onclick event each time the panel is re-rendered
                //  if we bind on document, the click event will only be triggered after the dialog.open one.
                JIRA.bind(JIRA.Events.NEW_CONTENT_ADDED, _.bind(function(e, context) {
                    this._initHandler(e, context, panelSelector, linkSelector);
                }, this));
                JIRA.bind("GH.DetailView.updated", _.bind(function(e) {
                    this._initHandler(e, AJS.$(panelSelector), panelSelector, linkSelector);
                }, this));

                // for single page view issue page only, bind directly to the links, as the links should be available at this time
                // if it's issue nav, the links would not be available at this time, thus no double binding
                $(linkSelector).on("simpleClick", function () {
                    var el = $(this);
                    instance._fireCreateBranchEvent(el);
                });

                // These three have to be bound on document as the target is outside of the context
                // Capture clicks on an instance in the cta dialog
                $(document).on("click", "#" + dialogId + " .jira-dialog-content .target", function () {
                    var el = $(this);
                    instance.fireCreateBranchEvent("dialog.click", {
                        title : el.find(".title").attr("title"),
                        applicationType : el.data("applicationType")
                    });
                });

                // Capture clicks on the cancel button in the cta dialog
                $(document).on("click", "#" + dialogId + " .jira-dialog-content button.cancel", function () {
                    instance.fireCreateBranchEvent("dialog.cancel");
                });

                // Capture clicks on the learn more link in the helptip
                $(document).on("click", ".cta-tooltip .cta-learn-more-link", function () {
                    JIRA.DevStatus.LearnMoreAnalytics.fireLearnMoreEvent("click.ctahelptipbranching");
                });
            }
        }
    };

    JIRA.DevStatus.AdminAnalytics = {
        fireAdminEvent : function (name, opts) {
            return JIRA.DevStatus.Analytics.fireEvent("admin." + name, opts);
        },

        initialize : function () {
            if (isAnalyticsEnabled()) {
                var instance = this,
                    configPanel = $("#project-config-panel-dev-status");
                // Capture clicks on edit permission link
                configPanel.find(editPermissionLinkSelector).on("click", function () {
                    instance.fireAdminEvent(
                            "editpermissionlink.config." +
                                ($("ul.permissions-list li.devstatus-admin-permission-item").length > 0 ?
                                        "withpermissions" : "nopermissions"));
                });
                // Capture clicks on appvertisement link
                configPanel.find(".learn-more-link").on("click", function () {
                    JIRA.DevStatus.LearnMoreAnalytics.fireLearnMoreEvent("click.adminappvertisementlink");
                });
                // Capture clicks on the conenct button dropdown
                configPanel.find("#project-config-applinks-connect").on("click", function () {
                    instance.fireAdminEvent("connect");
                });
                // Capture clicks on the individual connect buttons
                configPanel.find(".devstatus-admin-connect-button-option").on("click", function () {
                    var key = $(this).data("key") || "unknown";
                    instance.fireAdminEvent("connect." + key);
                });
            }
        }
    };

    JIRA.DevStatus.AdminSummaryAnalytics = {
        initialize : function () {
            if (isAnalyticsEnabled()) {
                // Capture clicks on edit permission link
                // always no permission, because the link would only appear when no permissions
                $("#project-config-webpanel-devstatus-admin-summary-panel").find(editPermissionLinkSelector).on("click", function () {
                    JIRA.DevStatus.AdminAnalytics.fireAdminEvent("editpermissionlink.summary.nopermissions");
                });
            }
        }
    };

})(AJS.$, JIRA);