define('layout/fly-out/view', ['zepto', 'backbone'], function ($, Backbone) {

    var noop = function () {};

    var FlyOutView = Backbone.View.extend({

        onAnimationFinished: noop,

        visible: false,
        expanded: false,
        changingVisibilityState: false,
        isAndroid: $('body').hasClass('platform-android'),

        classes: {
            OPEN : "fly-out-open",
            FULL_WIDTH: "fly-out-full-width",
            HIDING: "fly-out-hiding",
            SHOWING: "fly-out-showing",
            SHOWING_FINISHED: "fly-out-finished-showing",
            HIDING_FINISHED: "fly-out-finished-hiding"
        },

        $body: null,

        initialize: function (options) {
            // TODO: Why aren't these standard events?
            this.onAnimationFinished = options.onAnimationFinished || noop;
            this.$body = $(options.body || 'body');

            var that = this;
            this.$body.bind($.fx.transitionEnd, function (event) {
                that._restoreState(that);
            });
        },

        onUpdate: function() {
            if (this.isAndroid) {
                // Since android will never throw the webkitTransitionEnd event, 
                // we instead mimic that behaviour after the flyout is updated
                this._restoreState(this);
            }
        },

        _restoreState: function(target) {
            // Updates the state of things after the flyout
            // opens or closes
            if (target.visible) {
                target.$body
                    .removeClass(target.classes.SHOWING)
                    .addClass(target.classes.SHOWING_FINISHED);

            } else {
                target.$body
                    .removeClass([target.classes.HIDING, target.classes.SHOWING].join(" "))
                    .addClass(target.classes.HIDING_FINISHED);
            }

            target.changingVisibilityState = false;

            target.onAnimationFinished();
        },

        _update: function () {
            // CSS Transforms handle all the animation.
            this.$body
                .toggleClass(this.classes.OPEN , this.visible)
                .toggleClass(this.classes.SHOWING, this.changingVisibilityState && this.visible)
                .toggleClass(this.classes.HIDING, this.changingVisibilityState && !this.visible)
                .toggleClass(this.classes.FULL_WIDTH, this.expanded);
            $('.header-button.flyout-button').toggleClass('active', this.visible);

            if (this.changingVisibilityState) {
                this.$body.removeClass(this.classes.SHOWING_FINISHED + " " + this.classes.HIDING_FINISHED);
            }

            this.onUpdate();
        },

        toggle: function () {
            this.visible = !this.visible;
            this.changingVisibilityState = true;
            this._update();
        },

        show: function () {
            if (this.visible) {
                return;
            }

            this.changingVisibilityState = true;
            this.visible = true;
            this._update();
        },

        hide: function () {
            if (!this.visible) {
                return;
            }

            this.changingVisibilityState = true;
            this.restoreWidth();
            this.visible = false;
            this._update();
        },

        expandWidth: function () {
            if (this.expanded) {
                return;
            }

            this.changingVisibilityState = false;
            this.expanded = true;
            this._update();
        },

        restoreWidth: function () {
            if (!this.expanded) {
                return;
            }

            this.changingVisibilityState = false;
            this.expanded = false;
            this._update();
        },

        isVisible: function () {
            return !!this.visible;
        },

        isChangingVisibilityState: function () {
            return !!this.changingVisibilityState;
        }
    });

    return FlyOutView;
});