define('feature/open-app-button/open-app-button',

    ['./open-app-button.less', './open-app-button.soy', 'zepto', 'backbone', 'util/app-data'],
    function (css, Templates, $, Backbone, appData) {

        const JIRA_IOS_APP_STORE_URL = 'https://itunes.apple.com/us/app/id1405353949?mt=8';
        const JIRA_ANDROID_PLAY_STORE_URL = 'market://details?id=com.atlassian.jira.server';

        const APP_CUSTOM_URL_SCHEME_PROTOCOL = 'jira-server:';
        const NAVIGATION_SOURCE = 'open-in-app-button';

        /**
         * Pages that this button supports for deep linking.
         * These values should be used as url parameters so that the deep linking context can be
         * resolved from the client side.
         * @type {{ISSUE: string, FILTERS: string}}
         */
        const Types = {
            ISSUE: 'issue',
            FILTERS: 'filters'
        };

        /**
         * Provides support for creating a custom URL Scheme for deep-linking to the mobile app.
         * The url will be represented by the format:
         * url?param1=value1&param2=value2...
         * @param url The base customURLScheme, e.g. jira-server://server-base-url
         * @constructor
         */
        function CustomUrlScheme(url) {
            this.url = url;

            this._addParam = function (key, value) {
                const paramString = [encodeURI(key), encodeURI(value)].join('=');
                const joinChar = this.containsParams() ? '&' : '?';
                this.url = [this.url, paramString].join(joinChar);
            };

            this.addIssue = function(value) {
                this._addParam('type', Types.ISSUE);
                this._addParam('value', value);
            };

            this.addFilters = function() {
                this._addParam('type', Types.FILTERS);
            };

            this.addSource = function() {
                this._addParam('source', NAVIGATION_SOURCE);
            };

            this.containsParams = function() {
                return this.url.indexOf('?') > -1;
            };

            this.addSource(); // Add the source on initialization
        }

        /**
         * View for the "Open in app" button
         * Depends on having either of the two parameters issueKey or jql
         * issueKey: Required if the button is being rendered via the issue-view
         * jql: Required if the button is being rendered via the issue-list-view
         */
        return Backbone.View.extend({

            id: 'app-banner',

            events: {
                'click .app-open-button': 'openApp'
            },

            render: function () {
                const shouldShowAppButton = appData.get('show-open-app-button');
                if (shouldShowAppButton === 'true') {
                    this.$el.html(Templates.drawButton({
                        page: this.options.page
                    }));
                    $('#outer-container').append(this.$el);
                }
            },

            openApp: function () {
                const mobileCustomUrlScheme = this.getMobileCustomUrlScheme(this.options.issueKey);
                const appStoreUrl = this.getAppStoreUrl();
                window.location = mobileCustomUrlScheme;

                // This is a temporary solution to avoid an issue with iOS 12.4 and above where
                // Safari treats the app store as a trusted app and does not alert the user before delegation.
                if (!this.isIos()) {
                    // Due to privacy issues we cannot detect if the app is installed and instead
                    // must rely on this timeout block execution if the customUrlScheme was not resolved.
                    window.setTimeout(function () {
                        window.location = appStoreUrl;
                    }, 200);
                }
            },

            /**
             * Determines the expected customUrlScheme to use based on:
             * - If the issueKey is present then it should attempt to open the corresponding issue within the app.
             * - If the issueKey is not present then it should attempt to open the filter list.
             * @param issueKey Issue key of the issue being viewed (from issue-view).
             * @returns {string} The customUrlScheme expected by the app.
             */
            getMobileCustomUrlScheme: function (issueKey) {
                const urlString = appData.get('base-url').replace(/http:|https:/, APP_CUSTOM_URL_SCHEME_PROTOCOL);
                const customUrlScheme = new CustomUrlScheme(urlString);

                if (issueKey) {
                    customUrlScheme.addIssue(issueKey);
                } else {
                    customUrlScheme.addFilters();
                }
                return customUrlScheme.url;
            },

            getAppStoreUrl: function () {
                return this.isIos() ? JIRA_IOS_APP_STORE_URL : JIRA_ANDROID_PLAY_STORE_URL;
            },

            /**
             * References the userAgentString value of the device's OperatingSystemFamily
             */
            isIos: function () {
                const iosFamily = ['iPhone', 'iPad'];
                return iosFamily.indexOf(appData.get('os')) > -1;
            }
        });


    }
);
