/**
 * @module quick-edit/form/helper/helper
 */
define('quick-edit/form/helper/helper', [
    'quick-edit/util/jira',
    'quick-edit/form/model',
    'quick-edit/util/class',
    'jira/jquery/plugins/isdirty',
    'jira/util/events',
    'jira/dialog/dialog',
    'underscore',
    'jquery'
], function(
    JiraUtil,
    FormsModel,
    Class,
    DirtyForm,
    Events,
    Dialog,
    _,
    jQuery
){
    'use strict';

    var escapeHtml = AJS.escapeHTML;

    /**
     * A helper class to share between Configurable and Unconfigurable forms
     *
     * @class CreateIssueFormHelper
     * @extends Class
     */
    return Class.extend({
        init: function (form) {
            this.form = form
        },

        /**
         * If the create another checkbox is checked will notify the user of newly created issue and clear the form
         * (Except for pinned values). Otherwise, directs the browser to the newly created issue.
         *
         * @param data
         * ... {String} issueKey
         */
        handleSubmitSuccess: function (data) {
            function issueCreatedMessage(issueKey, summary, url) {
                var link = '<a href="' + url + '">' + issueKey + ' - ' + escapeHtml(summary) + '</a>';
                return AJS.I18n.getText('issue.create.created', link);
            }

            var instance = this,
                summary = this.form.$element.find("#summary").val();

            data.summary = summary;

            this.form.triggerEvent("submitted", [data, this]);
            this.form.triggerEvent("issueCreated", [data, this], true);

            if (instance.form.model.isInMultipleMode()) {

                this.form.render().done(function () {
                    var message = issueCreatedMessage(data.issueKey, summary, contextPath + "/browse/" + data.issueKey);
                    JiraUtil.applySuccessMessageToForm(instance.form.getForm(), message);
                });

            } else {
                this.form.triggerEvent("sessionComplete");
            }
        },

        /**
         * Gets checkbox that when checked allows multiple issue creating
         *
         * @return jQuery
         */
        getCreateAnotherCheckbox: function () {
            return this.form.$element.find("#qf-create-another");
        },

        /**
         * Determines if we are creating more entries.
         *
         * @return Boolean
         */
        isInMultipleMode: function () {
            return this.getCreateAnotherCheckbox().is(":checked");
        },

        /**
         * Gets project <select>
         *
         * @return {jQuery}
         */
        getProjectField: function () {
            return this.form.$element.find(".issue-setup-fields #project");
        },

        /**
         * Gets issueType <select>
         *
         * @return {jQuery}
         */
        getIssueTypeField: function () {
            return this.form.$element.find(".issue-setup-fields #issuetype");
        },

        /**
         * Serializes form and specifies which form values to retain. This is different to a normal serialization as
         * it includes any other fields set during the session. JRADEV-9466 e.g you can set a priority then specify a project that hasn't
         * got a priority field then come back to the original project and the priority will still be remembered.
         *
         * @return String
         */
        serializeForToggle: function (projectHasChanged) {

            if (!this.form.model.hasRetainFeature()) {
                return this.form.getForm().serialize();
            }

            // All fields (and their values) that were ever displayed to the user, before switching to this project/issuetype screen
            var prevActiveFields = this.form.model.prevActiveFields || [];

            // All fields (and their values) on the current project/issuetype screen. Including those not visible.
            var currentSerialization = this.form.getForm().serializeArray();

            // a shorthand
            var normalizeParam = function(param) {
                return this.form.model.normalizeFieldParam(param.name, param.value);
            }.bind(this);

            // The ids of the fields that are currently visible.
            var currentActiveIds = this.form.getActiveFieldIds();

            // An array of all the active fields that need to be passed onto the next form
            var currentActive = jQuery.grep(currentSerialization, function (param) {
                return jQuery.inArray(normalizeParam(param), currentActiveIds) !== -1;
            });

            // A filtered list of the previously active fields that are NOT visible on the current screen
            var filteredPreviousActive = jQuery.grep(prevActiveFields, function (param) {
                return jQuery.inArray(normalizeParam(param), currentActiveIds) === -1;
            });

            // A filtered list of the previously active field ids that are NOT visible on the current screen
            var filteredPrevActiveIds = jQuery.map(filteredPreviousActive, function (param) {
                return normalizeParam(param);
            });

            // A combined list of the previously active and currently active. Representing what should be retained when we
            // switch issue types
            var combinedActive = jQuery.merge(currentActive, filteredPreviousActive);

            // A combined list of the previously active and currently active. Representing what should be retained when we
            // switch issue types
            var combinedActiveIds = jQuery.merge(currentActiveIds, filteredPrevActiveIds);

            // Update the history of active fields
            this.form.model.prevActiveFields = jQuery.extend(true, [], combinedActive);

            // If the project has changed, then execute any actions specific to project changes.
            if (projectHasChanged) {
                combinedActiveIds = this.executeProjectChangedSecuritySettingAction(combinedActiveIds);
            }

            // do not retain custom field values that are not dirty, so theirs values may get default from another context
            combinedActiveIds = combinedActiveIds.filter(function(id) {
                // either it's not custom field or it is dirty
                return !this.form.model.isCustomFieldId(id) || this.form.model.fieldIsDirty(id);
            }.bind(this));

            // Prepare the list of paramaters we will post to the server
            var postParams = jQuery.merge([{name: "retainValues", value: true}], combinedActive);

            // Don't forget project and issue type. We always need this
            jQuery.each(currentSerialization, function (i, param) {
                if (param.name === "pid" || param.name === "issuetype") {
                    postParams.push(param)
                }
            });

            // So the server knows not to clear retained fields
            postParams.push({name: "toggle", value: true});

            // Let the server know that we want to retain all the values we send it. So the next screen will have the
            // pre populated values
            jQuery.each(combinedActiveIds, function (i, id) {
                postParams.push({name: "fieldsToRetain", value: id});
            });

            postParams.push({name: "formToken", value: this.form.model.getFormToken()});

            // Finally format the data into a post body.
            return jQuery.param(postParams)

        },

        /**
         * When a project changes, remove the 'security' setting from those that are retained to
         * force it to set the default.
         */
        executeProjectChangedSecuritySettingAction: function (fieldsToSendIds) {
            return _.reject(fieldsToSendIds, function(v) { return v === "security"; });
        },

        /**
         * Prepare any fields to be retained.  This incorporates some logic
         * on fields that should not be retained and should take their default values
         * when projects or issue types change.
         */
        prepareFieldsToRetain: function (combinedActiveIds, postParams) {
            jQuery.each(combinedActiveIds, function (i, id) {
                postParams.push({name: "fieldsToRetain", value: id});
            });

            return postParams;
        },

        /**
         * Sets issue type and updates form (by rerendering it) with the correct fields
         *
         * @param {String} issueType
         */
        setIssueType: function () {
            var instance = this,
                serialisedForm = this.serializeForToggle();
            this.form.invalidateFieldsCache(); // invalidate fields cache
            this.form.disable();
            this.form.render(serialisedForm).done(function () {
                instance.form.enable();
            });
        },


        /**
         * Sets projectId and updates form (by rerendering it) with the correct fields
         *
         * @param projectId
         */
        setProjectId: function () {
            var instance = this,
                serialisedForm = this.serializeForToggle(true);
            this.form.invalidateFieldsCache(); // invalidate fields cache
            this.form.disable();
            this.form.render(serialisedForm).done(function () {
                instance.form.enable();
            });
        },

        /**
         * Checks if the form is currently rendered in a dirty state, so this state could be preserved after changing the issue type
         * See: https://jira.atlassian.com/browse/JRA-38151
         */
        saveFormDirtyStatus: function () {
            this.form.isDirty = this.form.getForm().isDirty();
        },

        /**
         * Restores the dirty status after the new issue type form is rendered
         * See: https://jira.atlassian.com/browse/JRA-38151
         */
        restoreFormDirtyStatus: function () {
            if( this.form.isDirty ) {
                this.form.getForm().addClass(DirtyForm.ClassNames.BY_DEFAULT);
            }
        },

        /**
         * Decorates form with events
         */
        decorate: function () {
            var instance = this;

            this.getProjectField().change(function () {
                instance.saveFormDirtyStatus();
                Events.trigger("dialogBeforeContentChange",["projectId", this.value, Dialog.current.options.id]);
                instance.setProjectId(this.value);
            });

            this.getIssueTypeField().change(function () {
                instance.saveFormDirtyStatus();
                Events.trigger("dialogBeforeContentChange",["issueType", this.value, Dialog.current.options.id]);
                instance.setIssueType(this.value);
            });

            this.getCreateAnotherCheckbox().change(function () {
                instance.form.model.setIsMultipleMode(this.checked);
            });
        },

        /**
         * Decorate form fields with events
         */
        decorateFields: function() {
            var instance = this;

            // clear listeners from our namespace and bind them again
            this.form.$element.find('*[id^="customfield_"]').
                off("change.customField").
                on("change.customField", function() {
                    if (instance.form.model.isCustomFieldId(this.name)) {
                        instance.form.model.markFieldAsDirty(this.name);
                    }
                });
        },

        sendQuickEditAnalyticsEvent: function(args) {
            var instance = args.instance;
            var name = args.name;
            var customFields;
            var field;

            var allFormFields = instance.model.fields.map(function(field) {
                return {
                    id: field.id,
                    required: field.required
                };
            });

            if (args.field) {
                field = {
                    id: args.field.descriptor.id,
                    required: args.field.descriptor.required
                }
            }

            instance.model.getActiveFieldIds().done(function(ids) {
                customFields = allFormFields.filter(function(field) {
                    return ids.some(function(fieldId) {
                        return fieldId === field.id;
                    });
                });

                AJS.trigger('analyticsEvent', {
                    name: name,
                    data: {
                        allFieldsShown: args.allFieldsShown,
                        field: JSON.stringify(field),
                        allFields: JSON.stringify(allFormFields),
                        displayedFields: args.allFieldsShown ? JSON.stringify(allFormFields) :
                          JSON.stringify(customFields)
                    }
                });
            });
        }

    });
});

/**
 * @deprecated JIRA.Forms.CreateIssueHelper
 */
AJS.namespace('JIRA.Forms.CreateIssueHelper', null, require('quick-edit/form/helper/helper'));
