/**
 * @module quick-edit/form/configurable/edit-issue
 */
define('quick-edit/form/configurable/edit-issue', [
    'quick-edit/form/abstract/configurable',
    'quick-edit/util/jira',
    'quick-edit/templates/issue',
    'jira/util/users/logged-in-user',
    'wrm/context-path',
    'jquery'
], function(
    AbstractConfigurableForm,
    JiraUtil,
    IssueTemplates,
    User,
    contextPath,
    jQuery
){
    'use strict';

    /**
     * A View that renders a configurable version of the edit issue form. This version allows you to reorder fields and
     * add and remove fields.
     *
     * @class ConfigurableEditIssueForm
     * @extends AbstractConfigurableForm
     */
    return AbstractConfigurableForm.extend({
        /**
         * @constructor
         * @param options
         * ... {FormModel} model that gets fields and sets user preferences
         * ... {String, Number} issue id
         * ... {Array<String>} globalEventNamespaces - Events will be triggered on these namespaces
         * e.g EditForm.switchedToConfigurableForm. EditForm being the specified global namespace.
         */
        init: function (options) {
            this.model = options.model;
            this.globalEventNamespaces = options.globalEventNamespaces || [];
            this.issueId = options.issueId;
            this.title = options.title;

            // Form values will be serialized and POSTed to this url
            this.action = contextPath() + "/secure/QuickEditIssue.jspa?issueId=" + options.issueId + "&decorator=none";
            // The container element. All html will be injected into this guy.
            this.$element = jQuery("<div />").addClass("qf-form qf-configurable-form");
        },

        /**
         * Reloads window after form has been successfully submitted
         */
        handleSubmitSuccess: function (data) {
            this.triggerEvent("submitted", [data]);
            this.triggerEvent("issueEdited", [data], true);
            this.triggerEvent("sessionComplete");
        },

        /**
         * Determines if there are any visible fields. Comment field not included.
         *
         * @return Boolean
         */
        hasNoVisibleFields: function () {
            var deferred = jQuery.Deferred(),
                activeFieldIds = this.getActiveFieldIds();

            this.model.getUserFields().done(function (userFields) {
                if (userFields.length === 1 && userFields[0] === "comment") {
                    deferred.resolve(false);
                } else if (activeFieldIds.length === 0 || (activeFieldIds.length === 1 && activeFieldIds[0] === "comment")) {
                    deferred.resolve(true);
                } else {
                    deferred.resolve(false);
                }
            });

            return deferred.promise();
        },

        _render: function () {
            var deferred = jQuery.Deferred(),
                instance = this,
                html = IssueTemplates.editIssueForm({
                    issueId: this.issueId,
                    title: this.title,
                    atlToken: instance.model.getAtlToken(),
                    formToken: instance.model.getFormToken(),
                    isConfigurable: true,
                    modifierKey: JiraUtil.getModifierKey(),
                    showFieldConfigurationToolBar: !User.isAnonymous()
                });

            this.$element.html(html); // add form chrome to container element

            // render fields
            instance.renderFormContents().done(function (el, scripts) {
                instance.getFieldById("comment").done(function (field) {
                    instance.getFormContent().append(field.$element); // put comment field at end of form
                });
                deferred.resolveWith(instance, [instance.$element, scripts]);
            });

            return deferred.promise();
        }

    });
});

/**
 * @deprecated JIRA.Forms.ConfigurableEditIssueForm
 */
AJS.namespace('JIRA.Forms.ConfigurableEditIssueForm', null, require('quick-edit/form/configurable/edit-issue'));