/**
 * @module quick-edit/form/abstract/configurable
 */
define('quick-edit/form/abstract/configurable', [
    'quick-edit/form/abstract',
    'quick-edit/form/field/configurable',
    'quick-edit/form/helper/helper',
    'jquery'
], function(
    AbstractForm,
    ConfigurableField,
    Helper,
    jQuery
) {
    'use strict';

    /**
     * A View class that renders a form. The form provides controls that allows a user to configure which fields are shown
     * using a picker (@see FieldPicker). Users can also configure the order of these fields using
     * drag and drop.
     *
     * @class AbstractConfigurableForm
     * @extends AbstractForm
     */
    return AbstractForm.extend({

        /**
         * Gets all fields
         *
         * @param Array<ConfigurableField> fields
         * @return jQuery Promise
         */
        getFields: function () {
            var deferred = jQuery.Deferred(),
                instance = this;

            if (!this.fields) {
                this.fields = [];

                this.model.getConfigurableFields().done(function (fields) {
                    jQuery.each(fields, function (i, descriptor) {
                        var field = instance.createField(descriptor);
                        instance.fields.push(field);

                    });
                    deferred.resolveWith(instance, [instance.fields]);
                });
            } else {
                deferred.resolveWith(this, [this.fields]);
            }

            return deferred.promise();
        },

        /**
         * Gets ids for all visible fields
         * @return Array
         */
        getActiveFieldIds: function () {
            var ids = [],
                els = this.$element.find(".qf-field.qf-field-active:not(.qf-required), .qf-field.qf-required:not(.qf-field-active)");

            jQuery.each(els, function (i, el) {
                var $el = jQuery(el);
                var model = $el.data("model");
                var id = $el.data("field-id") || model && model.getId();

                // We get the id from the field control we attached using jQuery data.
                if(id != null) {
                    ids.push(id);
                }

                // Attachments are a special case because their checkboxes are added dynamically and are not part of the "model"
                if (id === "attachment") {
                    $el.find('input[name=filetoconvert]').each(function() {
                        ids.push(this.id);
                    });

                }
            });

            return ids;
        },

        /**
         * Creates Field View
         *
         * @param descriptor
         * @return {ConfigurableField}
         */
        createField: function (descriptor) {
            descriptor.hasVisibilityFeature = this.model.hasVisibilityFeature(descriptor);

            if (this.model.hasRetainFeature(descriptor)) {
                descriptor.hasRetainFeature = true;
                descriptor.retainValue = this.model.hasRetainedValue(descriptor);
            }

            var instance = this,
                field = new ConfigurableField(descriptor),
                helper = new Helper(instance);

            if (descriptor.hasVisibilityFeature) {
                // When we activate a field focus & persist it
                field.bind("activated", function () {
                    instance.model.setUserFields(instance.getActiveFieldIds());
                    field.highlight();
                    instance.triggerEvent("QuickForm.fieldAdded", [field]);

                    helper.sendQuickEditAnalyticsEvent({
                        name: 'quickeditform.custom.added',
                        field: field,
                        instance: instance
                    });
                }).bind("disabled", function () {
                    instance.model.setUserFields(instance.getActiveFieldIds());
                    instance.triggerEvent("QuickForm.fieldRemoved", [field]);

                    helper.sendQuickEditAnalyticsEvent({
                        name: 'quickeditform.custom.removed',
                        field: field,
                        instance: instance
                    });
                });
            }

            return field;
        },

        /**
         * Gets the field view instance by id
         *
         * @param id
         * @return jQuery.Promise
         */
        getFieldById: function (id) {
            var instance = this,
                deferred = jQuery.Deferred();

            this.getFields().done(function (fields) {
                jQuery.each(fields, function (i, field) {
                    if (field.getId() === id) {
                        deferred.resolveWith(instance, [field]);
                    }
                });

                deferred.rejectWith(instance, []);
            });

            return deferred.promise();
        },

        /**
         * Determines if there are any visible fields
         *
         * @return Boolean
         */
        hasNoVisibleFields: function () {
            var deferred = jQuery.Deferred();
            deferred.resolve(this.getActiveFieldIds().length === 0);
            return deferred.promise();
        },

        /**
         * Renders form contents and applies sortable control
         *
         * @return jQuery.promise
         */
        renderFormContents: function () {
            var deferred = jQuery.Deferred(),
                scripts = jQuery(),
                instance = this;

            instance.getFields().done(function (fieldsResult) {
                var fields = instance.reorderFieldsToRender(fieldsResult);
                instance.model.getActiveFieldIds().done(function (activeIds) {
                    jQuery.each(fields, function () {
                        var result = this.render();
                        // JRADEV-9069 Build up collection of all script tags to be executed post render
                        // Look at FormContainer.render for actual execution
                        scripts = scripts.add(result.scripts);
                        instance.getFormContent().append(result.element);
                    });

                    // append active fields in prescribe order first
                    jQuery.each(activeIds, function (i, fieldId) {
                        jQuery.each(fields, function () {
                            if (this.getId() === fieldId) {
                                this.activate(true);
                            }
                        });
                    });

                    // Now the inactive ones. We have to append as the field values need to be serialized. Also if there
                    // are any js controls they can be bound so that when we toggle the visibility they actually work.
                    jQuery.each(fields, function () {
                        if (!this.isActive()) {
                            this.disable(true);
                        }
                    });

                    // If we have no fields visible, append first 3 (JRADEV-6669)
                    instance.hasNoVisibleFields().done(function (answer) {
                        if (answer === true) {
                            for (var i=0; i < 3; i++) {
                                if (fields[i]) {
                                    fields[i].activate(true);
                                }
                            }
                        }

                        deferred.resolveWith(this, [instance.$element, scripts]);
                    });
                });
            });

            return deferred.promise();
        },

        /**
         * Reorder the fields which will be rendered onto form
         *
         * @param fields to reorder
         * @returns Array
         */
        reorderFieldsToRender: function(fields) {
            return fields;
        }
    });
});

/**
 * @deprecated JIRA.Forms.AbstractConfigurableForm
 */
AJS.namespace('JIRA.Forms.AbstractConfigurableForm', null, require('quick-edit/form/abstract/configurable'));